package edu.fing.image;

import java.io.*;
import java.awt.*;
import java.awt.image.*;

public class ImageMatrixFP
{
  public short planes;
  public int width;
  public int height;
  public double[][][] imageData;

  public ImageMatrixFP(int planes, int width, int height)
  {
    initialize(planes, width, height);
  }

  public ImageMatrixFP(String fileName)
  {
    loadBitmap(fileName);
  }

  public void initialize(int planes, int width, int height)
  {
    this.planes = (short) planes;
    this.width = width;
    this.height = height;
    imageData = new double[planes][height][width];
  }

  public BufferedImage getImage()
  {
    int[] buffer = new int[planes * height * width];
    int idx = 0;
    for(int j = 0; j < height; j++)
    {
      for(int i = 0; i < width; i++)
      {
        for(int k = 0; k < planes; k++)
        {
          buffer[idx++] = (int)imageData[k][j][i];
        }
      }
    }
    BufferedImage tempImage = new BufferedImage(width, height, planes == 1 ? BufferedImage.TYPE_BYTE_GRAY : BufferedImage.TYPE_INT_RGB);
    WritableRaster raster = tempImage.getRaster();
    raster.setPixels(0, 0, width, height, buffer);
    return tempImage;
  }


  ImageMatrixCanvas canvas;
  public ImageMatrixCanvas getCanvas()
  {
    if(canvas == null)
    {
      canvas = new ImageMatrixCanvas(this.toImageMatrix());
    }
    return canvas;
  }

  public ImageMatrix toImageMatrix()
  {
      ImageMatrix image = new ImageMatrix(planes, width, height);

// Busco limites
      for (int k = 0; k < planes; k++)
      {
          double max = -Double.MAX_VALUE;
          double min = Double.MAX_VALUE;
          for (int j = 0; j < height; j++)
          {
              for (int i = 0; i < width; i++)
              {
                  if (imageData[k][j][i] > max)
                  {
                      max = imageData[k][j][i];
                  }
                  if (imageData[k][j][i] < min)
                  {
                      min = imageData[k][j][i];
                  }
              }
          }

// Muevo rango a 0-255 (uniforme)
          double step = (max - min) / 255;
          System.out.println("ToImageMatrix: " + max + " - " + min + " - " + step);
          for (int j = 0; j < height; j++)
          {
              for (int i = 0; i < width; i++)
              {
                  image.imageData[k][j][i] = (short) ((imageData[k][j][i] - min) / step);
              }
          }
      }

      return image;
    }

  public void loadBitmap(String fileName)
  {
    Image image = null;
    try
    {
      if(fileName.toLowerCase().endsWith(".pgm") ||
               fileName.toLowerCase().endsWith(".ppm"))
      {
        image = PxMImage.loadPxM(fileName);
      }
      else
      {
        image = Toolkit.getDefaultToolkit().getImage(fileName);
      }
    } catch(IOException e)
    {
      System.err.println(e.toString());
    }
    if(image == null)
    {
      throw new Error("No se puede cargar la imagen: " + fileName);
    }
    initializeFromImage(image);
  }

  public void initializeFromImage(Image image)
  {
    if(image instanceof BufferedImage && ( (BufferedImage) image).getType() == BufferedImage.TYPE_BYTE_GRAY)
    { // Si se trata de una imagen en escala de grises
      BufferedImage bufferedImage = (BufferedImage) image;
      initialize(1, bufferedImage.getWidth(), bufferedImage.getHeight());
      Raster raster = bufferedImage.getData();
      int[] vector = new int[height * width];
      raster.getPixels(0, 0, width, height, vector);
      int idx = 0;
      for(int j = 0; j < height; j++)
      {
        for(int i = 0; i < width; i++)
        {
          imageData[0][j][i] = (double) ( (vector[idx++]) & 0xff);
        }
      }
    }
    else
    {
      Component observer = new Component()
      {};
      MediaTracker tracker = new MediaTracker(observer);
      tracker.addImage(image, 0);
      try
      {
        tracker.waitForID(0); // Espero hasta que la imagen este completamente disponible
      } catch(InterruptedException e)
      {
        ;
      }
      initialize(3, image.getWidth(observer), image.getHeight(observer));
      int[] vector = new int[width * height];
      PixelGrabber pixelGrabber = new PixelGrabber(image, 0, 0, width, height, vector, 0, width);
      try
      {
        pixelGrabber.grabPixels();
      } catch(InterruptedException e)
      {
        ;
      }
      int k = 0;
      for(int j = 0; j < height; j++)
      {
        for(int i = 0; i < width; i++)
        {
          int pixel = vector[k++];
          imageData[0][j][i] = (double) ( (pixel >> 16) & 0xff);
          imageData[1][j][i] = (double) ( (pixel >> 8) & 0xff);
          imageData[2][j][i] = (double) ( (pixel) & 0xff);
        }
      }
    }
  }

  public int getCantPixels()
  {
    return planes*width*height;
  }

  public ImageMatrixEnumeration getPixelEnumerator(int iterateType)
  {
    return new ImageMatrixEnumeration(this.toImageMatrix(), iterateType);
  }

  public static PixelIteratorFP getApplyFilterIterator()
  {
    return new PixelIteratorFP()
    {
      public void processPixel(ImageMatrixFP image, int k, int j, int i, double pixel)
      {
        image.imageData[k][j][i] = pixel;
      }
    };
  }

  public static final int ITERATE_RASTER  = 1 << 0;
  public static final int ITERATE_HILBERT = 1 << 1;
  public static final int ITERATE_BLOCKS  = 1 << 2;
  public static final int ITERATE_ARGS_MASK = ~((1<<10)-1);
  public static final int ITERATE_BLOCKS_SIZE_8   = 1<<10 | ImageMatrixFP.ITERATE_BLOCKS;
  public static final int ITERATE_BLOCKS_SIZE_16  = 1<<11 | ImageMatrixFP.ITERATE_BLOCKS;
  public static final int ITERATE_BLOCKS_SIZE_32  = 1<<12 | ImageMatrixFP.ITERATE_BLOCKS;
  public static final int ITERATE_BLOCKS_SIZE_64  = 1<<13 | ImageMatrixFP.ITERATE_BLOCKS;


  public void iterate(int iterateType, final PixelIteratorFP iterator)
  {
    switch(iterateType & ~ITERATE_ARGS_MASK)
    {
      case ITERATE_BLOCKS:
        iterateBlocks(iterator, iterateType & ITERATE_ARGS_MASK);
        break;
      case ITERATE_RASTER:
        iterateRaster(iterator);
        break;
      case ITERATE_HILBERT:
        iterateHilbert(iterator);
        break;
      default:
        throw new IllegalArgumentException("Unrecognized iterate type: " + iterateType);
    }
  }

  public void iterateCompose(int iterateTypeF, final int iterateTypeG, final int widthG, final int heightG, final PixelIteratorFP iterator)
  {
    final ImageMatrixFP tempImage = new ImageMatrixFP(1, widthG, heightG);
    ImageMatrixFP iterateImage = new ImageMatrixFP(planes, (int)((width+widthG-1)/widthG), (int)((height+heightG-1)/heightG));
    iterateImage.iterate(iterateTypeF, new PixelIteratorFP()
    {
      public void processPixel(ImageMatrixFP image, int k, int j, int i, double pixelValue)
      {
        final int fromY = j * heightG;
        final int fromX = i * widthG;
        final int curK = k;
        tempImage.iterate(iterateTypeG, new PixelIteratorFP()
        {
          public void processPixel(ImageMatrixFP image, int k, int j, int i, double pixelValue)
          {
            if(fromY + j >= height || fromX + i >= width)return;
            iterator.processPixel(ImageMatrixFP.this, k, fromY + j, fromX + i, imageData[curK][fromY + j][fromX + i]);
          }
        });
      }
    });
  }

  private void iterateBlocks(PixelIteratorFP iterator, int arg)
  {
    if(arg == 0)arg = ITERATE_BLOCKS_SIZE_8;
    int ITERATE_BLOCKS_SIZE = (arg >> 7);
    for(int k = 0; k < planes; k++)
    {
      for(int j = 0; j < (int)(height/ITERATE_BLOCKS_SIZE); j++)
      {
        for(int i = 0; i < (int)(width/ITERATE_BLOCKS_SIZE); i++)
        {
          for(int y = 0; y < ITERATE_BLOCKS_SIZE; y++)
          {
            for(int x = 0; x < ITERATE_BLOCKS_SIZE; x++)
            {
              iterator.processPixel(this, k, j*ITERATE_BLOCKS_SIZE + y, i*ITERATE_BLOCKS_SIZE + x, imageData[k][j*ITERATE_BLOCKS_SIZE + y][i*ITERATE_BLOCKS_SIZE + x]);
            }
          }
        }
      }
    }
  }

  private void iterateRaster(PixelIteratorFP iterator)
  {
    for(int k = 0; k < planes; k++)
    {
      for(int j = 0; j < height; j++)
      {
        for(int i = 0; i < width; i++)
        {
          iterator.processPixel(this, k, j, i, imageData[k][j][i]);
        }
      }
    }
  }

  private void iterateHilbert(PixelIteratorFP iterator)
  { // Hago una iteracion mediante una curva de Hilbert
    // Si la imagen no tiene dimensiones adecuadas 2^n x 2^n lo que queda
    // Lo recorro mediante raster scan

    int hilbertSize = (int)log2(Math.min(width, height));
    for(int curK = 0; curK < planes; curK++)
    {
      Point hilbertPos = new Point(0, 0);
      iterator.processPixel(this, curK, hilbertPos.y, hilbertPos.x, imageData[curK][hilbertPos.y][hilbertPos.x]);
      doHilbert(iterator, curK, hilbertPos, hilbertSize, HILBERT_UP);
      // Ahora completo lo que quede de la imagen
      int curSize = (int)Math.pow(2, hilbertSize);
      for(int j = curSize; j < height; j++)
      {
        for(int i = 0; i < width; i++)
        {
          iterator.processPixel(this, curK, j, i, imageData[curK][j][i]);
        }
      }
      for(int j = 0; j < curSize; j++)
      {
        for(int i = curSize; i < width; i++)
        {
          iterator.processPixel(this, curK, j, i, imageData[curK][j][i]);
        }
      }
    }
  }

  private static final int HILBERT_UP = 0;
  private static final int HILBERT_DOWN = 1;
  private static final int HILBERT_LEFT = 2;
  private static final int HILBERT_RIGHT = 3;

  private void doHilbert(PixelIteratorFP iterator, int curK, Point hilbertPos, int hilbertSize, int hilbertDir)
  {
    if(hilbertSize == 1)
    {
      switch(hilbertDir)
      {
        case HILBERT_LEFT:
          hilbertMove(iterator, curK, hilbertPos, HILBERT_RIGHT);
          hilbertMove(iterator, curK, hilbertPos, HILBERT_DOWN);
          hilbertMove(iterator, curK, hilbertPos, HILBERT_LEFT);
          break;
        case HILBERT_RIGHT:
          hilbertMove(iterator, curK, hilbertPos, HILBERT_LEFT);
          hilbertMove(iterator, curK, hilbertPos, HILBERT_UP);
          hilbertMove(iterator, curK, hilbertPos, HILBERT_RIGHT);
          break;
        case HILBERT_UP:
          hilbertMove(iterator, curK, hilbertPos, HILBERT_DOWN);
          hilbertMove(iterator, curK, hilbertPos, HILBERT_RIGHT);
          hilbertMove(iterator, curK, hilbertPos, HILBERT_UP);
          break;
        case HILBERT_DOWN:
          hilbertMove(iterator, curK, hilbertPos, HILBERT_UP);
          hilbertMove(iterator, curK, hilbertPos, HILBERT_LEFT);
          hilbertMove(iterator, curK, hilbertPos, HILBERT_DOWN);
          break;
      }
      /* switch */
    }
    else
    {
      switch(hilbertDir)
      {
        case HILBERT_LEFT:
          doHilbert(iterator, curK, hilbertPos, hilbertSize - 1, HILBERT_UP);
          hilbertMove(iterator, curK, hilbertPos, HILBERT_RIGHT);
          doHilbert(iterator, curK, hilbertPos, hilbertSize - 1, HILBERT_LEFT);
          hilbertMove(iterator, curK, hilbertPos, HILBERT_DOWN);
          doHilbert(iterator, curK, hilbertPos, hilbertSize - 1, HILBERT_LEFT);
          hilbertMove(iterator, curK, hilbertPos, HILBERT_LEFT);
          doHilbert(iterator, curK, hilbertPos, hilbertSize - 1, HILBERT_DOWN);
          break;
        case HILBERT_RIGHT:
          doHilbert(iterator, curK, hilbertPos, hilbertSize - 1, HILBERT_DOWN);
          hilbertMove(iterator, curK, hilbertPos, HILBERT_LEFT);
          doHilbert(iterator, curK, hilbertPos, hilbertSize - 1, HILBERT_RIGHT);
          hilbertMove(iterator, curK, hilbertPos, HILBERT_UP);
          doHilbert(iterator, curK, hilbertPos, hilbertSize - 1, HILBERT_RIGHT);
          hilbertMove(iterator, curK, hilbertPos, HILBERT_RIGHT);
          doHilbert(iterator, curK, hilbertPos, hilbertSize - 1, HILBERT_UP);
          break;
        case HILBERT_UP:
          doHilbert(iterator, curK, hilbertPos, hilbertSize - 1, HILBERT_LEFT);
          hilbertMove(iterator, curK, hilbertPos, HILBERT_DOWN);
          doHilbert(iterator, curK, hilbertPos, hilbertSize - 1, HILBERT_UP);
          hilbertMove(iterator, curK, hilbertPos, HILBERT_RIGHT);
          doHilbert(iterator, curK, hilbertPos, hilbertSize - 1, HILBERT_UP);
          hilbertMove(iterator, curK, hilbertPos, HILBERT_UP);
          doHilbert(iterator, curK, hilbertPos, hilbertSize - 1, HILBERT_RIGHT);
          break;
        case HILBERT_DOWN:
          doHilbert(iterator, curK, hilbertPos, hilbertSize - 1, HILBERT_RIGHT);
          hilbertMove(iterator, curK, hilbertPos, HILBERT_UP);
          doHilbert(iterator, curK, hilbertPos, hilbertSize - 1, HILBERT_DOWN);
          hilbertMove(iterator, curK, hilbertPos, HILBERT_LEFT);
          doHilbert(iterator, curK, hilbertPos, hilbertSize - 1, HILBERT_DOWN);
          hilbertMove(iterator, curK, hilbertPos, HILBERT_DOWN);
          doHilbert(iterator, curK, hilbertPos, hilbertSize - 1, HILBERT_LEFT);
          break;
      }
    }
  }

  private void hilbertMove(PixelIteratorFP iterator, int curK, Point hilbertPos, int hilbertDir)
  {
    switch(hilbertDir)
    {
      case HILBERT_UP:
        hilbertPos.translate(0, -1);
        break;
      case HILBERT_DOWN:
        hilbertPos.translate(0, 1);
        break;
      case HILBERT_LEFT:
        hilbertPos.translate(-1, 0);
        break;
      case HILBERT_RIGHT:
        hilbertPos.translate(1, 0);
        break;
    }
    iterator.processPixel(this, curK, hilbertPos.y, hilbertPos.x, imageData[curK][hilbertPos.y][hilbertPos.x]);
  }

  private double log2(double value)
  {
    return Math.log(value) / Math.log(2);
  }

  public ImageMatrixFP newSameSize()
  {
    return new ImageMatrixFP(planes, width, height);
  }

  public void showImage(String title)
  {
    ImageCanvas.showImage(title, toImageMatrix().getImage());
  }
}

