package com.brownsoft.ag;

import java.util.*;

/** Esta clase implementa los criterios de performance de DeJong y los aplica
 * como criterios de parada.
 * Al constructor se le pasa la escala de convergencia esperada
 * @see GeneracionSimple
 * @author Gustavo Brown
 * @version 1.0
 */
public class GeneracionSimpleDeJong extends GeneracionSimple
{
   protected double escala;
   protected int T;
   protected double onLinePerformance = -Double.MAX_VALUE;
   protected double offLinePerformance = -Double.MAX_VALUE;
   protected double bestFitnessSoFar = -Double.MAX_VALUE;
   protected double lastOffLinePerformance = -Double.MAX_VALUE;
   protected double lastOnLinePerformance = -Double.MAX_VALUE;

   /** Constructor
    * @param individuo Instancia de un individuo
    * @param pobSize tamao de la poblacin
    * @param pCross probabilidad de cruzamiento
    * @param pMutacion probabilidad de mutacin
    */
   public GeneracionSimpleDeJong(IIndividuo individuo, int pobSize, double pCross, double pMutacion) throws MotorException
   {
      this(individuo, pobSize, pCross, pMutacion, 1e8);
   }

   /** Constructor
    * @param individuo Instancia de un individuo
    * @param pobSize tamao de la poblacin
    * @param pCross probabilidad de cruzamiento
    * @param pMutacion probabilidad de mutacin
    * @param escala de convergencia (performance on-line) esperado
    */
   public GeneracionSimpleDeJong(IIndividuo individuo, int pobSize, double pCross, double pMutacion, double escala) throws MotorException
   {
      super(individuo, pobSize, pCross, pMutacion);
      this.escala = escala;
   }

   /** Indica que se finaliza la iteracion
    */
   public void finIteracion()
   {
      // Ahora calculo la on-line performance y la on-line performance de esta iteracion
      lastOnLinePerformance = onLinePerformance;
      T++;
      IIndividuo bestIndividuo = super.motor.getBestIndividuo();
      double currFitness = super.motor.getFitness(bestIndividuo);
      bestFitnessSoFar = (currFitness > bestFitnessSoFar ? currFitness : bestFitnessSoFar);
      onLinePerformance = (onLinePerformance * (T - 1) + currFitness) / T;
      offLinePerformance = (offLinePerformance * (T - 1) + bestFitnessSoFar) / T;
   }

   /** Obtiene la online performance
    * @return onLinePerformance
    */
   public double getOnLinePerformance()
   {
      return onLinePerformance;
   }

   /** Obtiene la offline performance
    * @return offLinePerformance
    */
   public double getOffLinePerformance()
   {
      return offLinePerformance;
   }

   /** Indica cuando debo finalizar las iteraciones
    * @return booleano indicando si debo terminar las iteraciones
    */
   public boolean getFinalizarIteracion()
   {
      if (T < 10)
      {
         return false; // Las primeras 10 iteraciones las hacemos siempre
      }
      return (onLinePerformance - lastOnLinePerformance) < escala;
   }

   double diff = 0;
}
