package com.brownsoft.ag;

import com.brownsoft.ag.fitnessScaler.FitnessScaler;
import java.util.*;

/** Esta clase implementa el algortmo gentico propiamente dicho
 * @version 1.0
 * @author Gustavo Brown
 */
public class MotorAG
{
   private Hashtable fitnessMapping = new Hashtable();
   private Vector poblacion = new Vector();
   private int totIteraciones;
   private IGeneracion generacion;
   private ISeleccion seleccion;
   private ICruzamiento cruzamiento;
   private IMutacion mutacion;
   private IMotorListener[] listeners = new IMotorListener[0];
   private FitnessScaler fitnessScaler = null;

   /** Constructor
    * @param generacion Genera la poblacion inicial
    * @param seleccion Operador de seleccion
    * @param cruzamiento Operador de cruzamiento
    * @param mutacion Operador de mutacion
    */
   public MotorAG(IGeneracion generacion, ISeleccion seleccion, ICruzamiento cruzamiento, IMutacion mutacion)
   {
      this.generacion = generacion;
      this.seleccion = seleccion;
      this.cruzamiento = cruzamiento;
      this.mutacion = mutacion;
      addListener(generacion);
      addListener(seleccion);
      addListener(cruzamiento);
      addListener(mutacion);

      // Primero obtengo la poblacion inicial
      generacion.inicializar(this);
      seleccion.inicializar(this);
      cruzamiento.inicializar(this);
      mutacion.inicializar(this, generacion.getpMutacion());
      totIteraciones = 0;
      poblacion = generacion.getPoblacionInicial();
   }

   /** Itera una vez el algoritmo
    * @return boolean true si la condicion de parada se ha cumplido
    */
   public boolean iterarOnce() throws MotorException
   {
      fireInicioIteracion();
      totIteraciones++;
      int poolSize = 0;
      int matingSize = generacion.getMatingSize();
      while (poolSize < matingSize)
      {
         // Selecciono individuos
         IIndividuo[] seleccionados = seleccion.seleccionar();

         // Los cruzo

         if (generacion.getpCross() >= PRNG.nextProbability())
         {
            seleccionados = cruzamiento.cruzar(seleccionados);
         }

         poolSize += seleccionados.length;

         // Aplico mutacin y los coloco en la poblacion intermedia
         for (int k = 0; k < seleccionados.length; k++)
         {
            seleccionados[k] = mutacion.mutar(seleccionados[k]);
         }
         seleccion.ponerEnPoolIntermedio(seleccionados);
      }
      poblacion = seleccion.generarNuevaPoblacion();
      fireFinIteracion();

      if (generacion.getFinalizarIteracion())
      {
         // Si se dio la condicion de parada del algoritmo disparo
         // el evento finAlgoritmo
         fireFinAlgoritmo();
         return true;
      }
      return false;
   }

   /** Itera una cantidad fija de veces el AG
    * @param cantIteraciones cantidad de veces a iterar
    */
   public void iterar(int cantIteraciones) throws MotorException
   {
      for (int i = 0; i < cantIteraciones; i++)
      {
         iterarOnce();
      }
   }

   /** Itera hasta que la condicion de parada se cumpla
    */
   public void iterar() throws MotorException
   {
      while (!generacion.getFinalizarIteracion())
      {
         iterarOnce();
      }
   }

   /** Obtiene la cantidad total de iteraciones realizadas por este motor
    * @return cantidad total de iteraciones
    */
   public int getTotIteraciones()
   {
      return totIteraciones;
   }

   /** Obtiene un Vector con la poblacion actual
    * @return Vector de IIndividuo con la poblacin actual
    */
   public Vector getPoblacion()
   {
      return poblacion;
   }

   /** Obtiene el Hashtable con los mapeos IIndividuo<-->Double(fitness)
    * @return Hashtable con los mapeos IIndividuo<-->Double(fitness)
    */
   public Hashtable getFitnessMapping()
   {
      return fitnessMapping;
   }

   /** Obtiene el fitness de un individuo sin escalar. Solo lo calcula si es necesario
    * @param individuo IIndividuo a obtener su fitness
    * @return double con el fitness del individuo sin escalar
    */
   public double getNonScaledFitness(IIndividuo individuo)
   {
      Double fitness = (Double) fitnessMapping.get(individuo);
      if (fitness == null)
      { // Si debo calcular el fitness
         fitness = new Double(individuo.getFitness());
         fitnessMapping.put(individuo.getCopy(), fitness);
      }
      return fitness.doubleValue();
   }

   /** Obtiene el fitness escalado de este individuo. Solo lo calcula si es necesario
    *  Para asignar un FitnessScaler se debe llamar a MotorAG.setFitnessScaler(IFitnessScaler)
    *  Si no hay ningun FitnessScaler asociado a este motor se retorna el fitness del individuo
    * @param individuo IIndividuo a obtener su fitness
    * @return double con el fitness escalado del individuo
    * @see MotorAG.setFitnessScaler
    * @see interfaz IFitnessScaler
    */
   public double getFitness(IIndividuo individuo)
   {
      if (fitnessScaler == null)
      {
         return getNonScaledFitness(individuo);
      }
      else
      {
         return fitnessScaler.getScaledFitness(individuo);
      }
   }

   /** Setea el FitnessScaler para este motor
    *  @param FitnessScaler a utilizar o null si no se desea utilizar ninguno
    */
   public void setFitnessScaler(FitnessScaler fitnessScaler)
   {
      this.fitnessScaler = fitnessScaler;
      if (fitnessScaler != null)
      {
         fitnessScaler.initScaler(this);
      }
   }

   /** Elimina el Hashtable de fitnesses, para que se vuelvan a calcular nuevamente
    */
   public void clearFitnesses()
   {
      fitnessMapping.clear();
   }

   /** Obtiene el mejor individuo de esta iteracin
    * @return el mejor Individuo de esta iteracin
    */
   public IIndividuo getBestIndividuo()
   {
      IIndividuo best = null;
      double bestFitness = -Double.MAX_VALUE;
      for (Enumeration enum = poblacion.elements(); enum.hasMoreElements(); )
      {
         IIndividuo individuo = (IIndividuo) enum.nextElement();
         double fitness = getFitness(individuo);
         if (fitness > bestFitness)
         {
            best = individuo;
            bestFitness = fitness;
         }
      }
      if (best == null)
      {
         return null;
      }
      return best.getCopy();
   }

   /** Agrega un listener de eventos del motor
    * @param listener El IMotorListener a agregar
    */
   public void addListener(IMotorListener listener)
   {
      IMotorListener[] newListeners = new IMotorListener[listeners.length + 1];
      System.arraycopy(listeners, 0, newListeners, 0, listeners.length);
      newListeners[listeners.length] = listener;
      listeners = newListeners;
   }

   /** Elimina un listener del motoro
    * @param listener El IMotorListener a eliminar
    */
   public void removeListener(IMotorListener listener)
   {
      for (int i = 0; i < listeners.length; i++)
      {
         if (listeners[i] == listener)
         {
            IMotorListener[] newListeners = new IMotorListener[listeners.length - 1];
            System.arraycopy(listeners, 0, newListeners, 0, i);
            System.arraycopy(listeners, i + 1, newListeners, i, listeners.length - i - 1);
            return;
         }
      }
   }

   /** Dispara el evento de inicio de iteracion
    */
   protected void fireInicioIteracion()
   {
      if (fitnessScaler != null)
      {
         fitnessScaler.fireInicioIteracion();
      }

      for (int i = 0; i < listeners.length; i++)
      {
         listeners[i].inicioIteracion();
      }
   }

   /** Dispara el evento de fin de iteracion
    */
   protected void fireFinIteracion()
   {
      for (int i = 0; i < listeners.length; i++)
      {
         listeners[i].finIteracion();
      }
   }

   /** Dispara el evento de fin del algoritmo
    */
   protected void fireFinAlgoritmo()
   {
      for (int i = 0; i < listeners.length; i++)
      {
         listeners[i].finAlgoritmo();
      }
   }

   /** Obtiene el IGeneracion de esta instancia
    *  @return IGeneracion de esta instancia
    */
   public IGeneracion getGeneracion()
   {
      return generacion;
   }

   /** Obtiene el operador de seleccion de esta instancia
    *  @return ISeleccion de esta instancia
    */
   public ISeleccion getOperadorSeleccion()
   {
      return seleccion;
   }

   /** Obtiene el operador de cruzamiento de esta instancia
    *  @return ICruzamiento de esta instancia
    */
   public ICruzamiento getOperadorCruzamiento()
   {
      return cruzamiento;
   }

   /** Obtiene el operador de mutacion de esta instancia
    *  @return IMutacion de esta instancia
    */
   public IMutacion getOperadorMutacion()
   {
      return mutacion;
   }
}
