package com.brownsoft.ag;

import com.brownsoft.ag.individuo.*;

/** Esta clase es utilizada ejecutar un operador genetico (cruzamiento y mutacin)
 * a individuos multiparmetros
 * Cada uno de los parametros puede tener un operador distinto
 * @see IndividuoMultiple
 * @author Gustavo Brown
 * @version 1.0
 */
public class OperadorMultiple implements ICruzamiento, IMutacion
{
   protected ICruzamiento[] cruzamientos;
   protected IMutacion[] mutaciones;
   protected MotorAG motor;

   /** Constructor
    * Crea una instancia de OperadorMultiple solamente con el operador de cruzamiento
    * @param cruzamientos Array de ICruzamiento con el operador de cruzamiento para cada parametro
    *
    */
   public OperadorMultiple(ICruzamiento[] cruzamientos)
   {
      this(cruzamientos, null);
   }

   /** Constructor
    * Crea una instancia de OperadorMultiple solamente con el operador de mutacion
    * @param mutaciones Array de IMutacion con el operador de mutacion para cada parametro
    *
    */
   public OperadorMultiple(IMutacion[] mutaciones)
   {
      this(null, mutaciones);
   }

   public OperadorMultiple(ICruzamiento[] cruzamientos, IMutacion[] mutaciones)
   {
      this.cruzamientos = cruzamientos;
      this.mutaciones = mutaciones;
      if (cruzamientos != null && mutaciones != null && cruzamientos.length != mutaciones.length)
      {
         throw new MotorError("La cantidad de elementos del array de cruzamientos difiera de la del array de mutaciones");
      }
   }

   /** Inicializa el operador de cruzamiento
    * @param motor MotorAG asociado
    */
   public void inicializar(MotorAG motor)
   {
      this.motor = motor;
      if (cruzamientos != null)
      {
         for (int i = 0; i < cruzamientos.length; i++)
         {
            cruzamientos[i].inicializar(motor);
         }
      }
   }

   /** Avisa que se inicia una nueva iteracin
    */
   public void inicioIteracion()
   {
      if (cruzamientos != null)
      {
         for (int i = 0; i < cruzamientos.length; i++)
         {
            cruzamientos[i].inicioIteracion();
         }
      }
      if (mutaciones != null)
      {
         for (int i = 0; i < mutaciones.length; i++)
         {
            mutaciones[i].inicioIteracion();
         }
      }
   }

   /** Indica que se finaliza la iteracion
    */
   public void finIteracion()
   {
      if (cruzamientos != null)
      {
         for (int i = 0; i < cruzamientos.length; i++)
         {
            cruzamientos[i].finIteracion();
         }
      }
      if (mutaciones != null)
      {
         for (int i = 0; i < mutaciones.length; i++)
         {
            mutaciones[i].finIteracion();
         }
      }
   }

   /** Avisa que finaliz la ejecucin del algoritmo
    */
   public void finAlgoritmo()
   {
      if (cruzamientos != null)
      {
         for (int i = 0; i < cruzamientos.length; i++)
         {
            cruzamientos[i].finAlgoritmo();
         }
      }
      if (mutaciones != null)
      {
         for (int i = 0; i < mutaciones.length; i++)
         {
            mutaciones[i].finAlgoritmo();
         }
      }
   }

   /** Cruza un grupo de individuos
    * @param seleccioandos grupo de individuos a cruzar
    * @return grupo de individuos cruzados
    * @exception InvalidCrossException en caso de que ocurra algun error
    */
   public IIndividuo[] cruzar(IIndividuo[] seleccionados) throws InvalidCrossException
   {
      // Primero chequeo que esta instancia de OperadorMultiple tenga definidos los operadores de cruzamiento
      if (cruzamientos == null)
      {
         throw new InvalidCrossException("No se definio el operador de cruzamiento para este OperadorMultiple");
      }

      // Ahora me aseguro que los individuos seleccionados sean instancias de IndividuoMultiple
      IndividuoMultiple[] seleccionadosMultiple = new IndividuoMultiple[seleccionados.length];
      try
      {
         for (int i = 0; i < seleccionados.length; i++)
         {
            seleccionadosMultiple[i] = (IndividuoMultiple) seleccionados[i];
         }
      }
      catch (ClassCastException e)
      {
         throw new MotorError("Solo se puede aplicar el OperadorMultiple a instancias de IndividuoMultiple", e);
      }

      IndividuoMultiple[] cruzadosMultiple = new IndividuoMultiple[seleccionados.length];
      for (int i = 0; i < seleccionados.length; i++)
      {
         cruzadosMultiple[i] = (IndividuoMultiple) seleccionadosMultiple[i].getCopy();
         motor.getFitnessMapping().remove(seleccionadosMultiple[i]);
      }

      for (int i = 0; i < cruzamientos.length; i++)
      {
         // Ahora formo un array de IIndividuo con el parametro i-esimo de cada IndividuoMultiple seleccionado
         IIndividuo[] seleccionadosParametro = new IIndividuo[seleccionados.length];
         for (int j = 0; j < seleccionados.length; j++)
         {
            seleccionadosParametro[j] = seleccionadosMultiple[j].getParameter(i);
         }
         // Ahora aplico el operador de cruzamiento a dichos individuos
         seleccionadosParametro = cruzamientos[i].cruzar(seleccionadosParametro);

         // Y ahora coloco el resultado
         for (int j = 0; j < seleccionados.length; j++)
         {
            cruzadosMultiple[j].setParameter(i, seleccionadosParametro[j]);
         }
      }
      return cruzadosMultiple;
   }

   /** Inicializa el operador de mutacion
    * @param motor MotorAG asociado
    * @param pMutacion probabilidad de mutacion
    */
   public void inicializar(MotorAG motor, double pMutacion)
   {
      if (mutaciones != null)
      {
         for (int i = 0; i < mutaciones.length; i++)
         {
            mutaciones[i].inicializar(motor, pMutacion);
         }
      }
   }

   /** Muta un individuo
    * @param individuo a mutar
    */
   public IIndividuo mutar(IIndividuo individuo) throws InvalidMutationException
   {
      // Primero chequeo que esta instancia de OperadorMultiple tenga definidos los operadores de cruzamiento
      if (mutaciones == null)
      {
         throw new InvalidMutationException("No se definio el operador de mutacion para este OperadorMultiple");
      }

      // Ahora me aseguro que los individuos seleccionados sean instancias de IndividuoMultiple
      if (! (individuo instanceof IndividuoMultiple))
      {
         throw new MotorError("Solo se puede aplicar el OperadorMultiple a instancias de IndividuoMultiple");
      }

      IndividuoMultiple individuoMultiple = (IndividuoMultiple) individuo;
      IndividuoMultiple mutadoMultiple = (IndividuoMultiple) individuoMultiple.getCopy();
      motor.getFitnessMapping().remove(individuoMultiple);

      for (int i = 0; i < mutaciones.length; i++)
      {
         mutadoMultiple.setParameter(i, mutaciones[i].mutar(individuoMultiple.getParameter(i)));
      }
      return mutadoMultiple;
   }
}
