package com.brownsoft.ag.fitnessScaler;

import com.brownsoft.ag.*;
import com.brownsoft.ag.individuo.*;
import java.util.*;

/** Este scaler implementa un algoritmo de nichos paralelos, como se describe
 * en el libro de Goldberg(capitulo 5)
 * El fitness se escala mediante la siguiente frmula
 * fitness_escalado = fitness_original/ Sum(s(d(xi, xj))
 * donde Sum denota sumatoria
 * 			 d indica distancia entre 2 individuos
 *       s indica la 'sharing function' que en este caso es triangular
 * La funcion de sharing triangular es como sigue:
 * 		sharing(distancia) = (sigmaShare - distancia) / sigmaShare
 * y para distancias mayores que sigmaShare la funcion de sharing da 0
 * Este scaler solo funciona con individuos del tipo bitString
 * @author Gustavo Brown
 * @version 1.0
 */
public class FitnessScalerEspeciacion extends FitnessScaler
{
   Hashtable sumSigmaShare = new Hashtable();
   double sigmaShare;

   /** Constructor
    *  @param parent El parent de este fitness scaler (o FitnessScalerSink)
    *  @param sigmaShare el sigma de la funcion de sharing medido en bits de distancia
    */
   public FitnessScalerEspeciacion(IFitnessScaler parent, double sigmaShare)
   {
      super(parent);
      this.sigmaShare = sigmaShare;
   }

   /** Este metodo debe ser definido por las subclases de FitnessScaler
    *  @return fitness escalado para que sea siempre positivo en este punto de la cadena
    */
   protected double scaleFitness(double fitness)
   {
      if (getProcessingIndividual() == null)
      {
         return fitness;
      }
      if (!sumSigmaShare.contains(getProcessingIndividual()))
      {
         setupSigmaShare( (IndividuoBitStreamSimple) getProcessingIndividual());
      }
      return fitness / ( (Double) sumSigmaShare.get(getProcessingIndividual())).doubleValue();
   }

   /** Obtiene la distancia entre 2 individuos */
   protected int getDistance(IndividuoBitStreamSimple individuo1, IndividuoBitStreamSimple individuo2)
   {
      long value1 = ( (Long) individuo1.getValue()).longValue();
      long value2 = ( (Long) individuo2.getValue()).longValue();
      int distance = 0;
      double dist = 1 / (double) individuo1.getBitWidth();
      for (int i = individuo1.getBitWidth(); i > 0; i--)
      {
         if ( (value1 & 1) != (value2 & 1))
         {
            distance += dist;
         }
         dist *= 2;
         value1 >>= 1;
         value2 >>= 1;
      }
      return distance;
   }

   /** Obtiene el sharing (es decir, le aplica la funcion de sharing a la distancia)
    * La implementacion por defecto es un sharing triangular, pero se puede derivar esta
    * clase para cambiar la funcin
    */
   protected double getSharing(int distance)
   {
      if (distance > sigmaShare)
      {
         return 0;
      }
      else
      {
         return (sigmaShare - distance) / sigmaShare;
      }
   }

   /** Calcula el sigmaShare para un individuo */
   protected void setupSigmaShare(IndividuoBitStreamSimple individuo)
   {
      double sharing = 0;
      for (Enumeration enum = super.getMotor().getPoblacion().elements(); enum.hasMoreElements(); )
      {
         IndividuoBitStreamSimple individuo2 = (IndividuoBitStreamSimple) enum.nextElement();
         int distance = getDistance(individuo, individuo2);
         sharing += getSharing(distance);
      }
      // Ahora guardo el Sum de los sharings en la tabla de hash
      sumSigmaShare.put(individuo.getCopy(), new Double(sharing));
   }

   /** Avisa que se inicia una nueva iteracin
    */
   public void inicioIteracion()
   {
      sumSigmaShare.clear();
      try
      {
         for (Enumeration enum = super.getMotor().getPoblacion().elements(); enum.hasMoreElements(); )
         {
            IndividuoBitStreamSimple individuo = (IndividuoBitStreamSimple) enum.nextElement();
            setupSigmaShare(individuo);
         }
      }
      catch (ClassCastException e)
      {
         throw new MotorError("El Scaler de especiacion solo puede trabajar con individuos del tipo IndividuoBitStreamSimple", e);
      }
   }
}