package com.brownsoft.ag.fitnessScaler;

import com.brownsoft.ag.*;
import java.util.*;

/** Este scaler se asegura que al momento de iniciar una iteracin, se ajusten los parametros
 *  para que todos los fitness de la poblacion actual sean positivos.
 *  En el caso de que en el medio de una iteracion (por ejemplo al realizar la mutacion el fitness
 *  escalado fuese a dar negativo, se retorna 0).
 *  Mediante este scaler se puede utilizar sin problemas los operadores de seleccion como el de la Ruleta
 *  que tienen como precondicion que los fitnesses NO sean negativos
 *  @Nota: Este scaler garantiza que los fitness escalados seran positivos HASTA este eslabon de la cadena
 *         Si luego este fitness le sigue otro scaler en la cadena, puede que el resultado no sea positivo!
 *        (por ejemplo, si tengo new FitnessScalerAxB(new FitnessScalerPositive(new FitnessScalerSink)), -1,0));
 *        en este caso el scaler AxB va a multiplicar por -1 el resultado, volviendolo negativo...
 * @author Gustavo Brown
 * @version 1.0
 */

public class FitnessScalerPositive extends FitnessScaler
{
   private double cantToSum;
   private boolean calculandoCantToSum = false;

   /** Constructor
    *  @param parent El parent de este fitness scaler (o FitnessScalerSink)
    */
   public FitnessScalerPositive(IFitnessScaler parent)
   {
      super(parent);
      cantToSum = 0;
   }

   /** Este metodo debe ser definido por las subclases de FitnessScaler
    *  @return fitness escalado para que sea siempre positivo en este punto de la cadena
    */
   protected double scaleFitness(double fitness)
   {
      if (calculandoCantToSum)
      { // Si estoy calculando la cantidad a sumar, no debo escalar
         return fitness;
      }

      if (fitness + cantToSum < 0)
      { // Si el fitness escalado de este individuo va a ser menor que 0, retorno 0
         return 0;
      }
      else
      { // Retorno el fitness escalado
         return fitness + cantToSum;
      }
   }

   /** Avisa que se inicia una nueva iteracin
    *  Aqui es donde calculo la cantidad a sumar de acuerdo a
    *  la poblacion actual
    */
   public void inicioIteracion()
   {
      calculandoCantToSum = true;
      double currentMinFitness = Double.MAX_VALUE;
      for (Enumeration enum = super.getMotor().getPoblacion().elements(); enum.hasMoreElements(); )
      {
         // Obtengo el fitness del individuo escalado hasta este eslabn de la cadena (es decir,
         // desde aqui hasta el Sink).
         // @Nota: Las operaciones que se hagan sobre el fitness desde este eslabon en adelante
         //        no son consideradas
         double currentFitness = super.getScaledFitness( (IIndividuo) enum.nextElement());
         if (currentFitness < currentMinFitness)
         {
            currentMinFitness = currentFitness;
         }
      }

      // Ahora veo si tengo que setear el cantToSum para que los fitnesses sean todos positivos
      if (currentMinFitness < 0)
      {
         cantToSum = -currentMinFitness;
      }
      else
      {
         cantToSum = 0;
      }
      calculandoCantToSum = false;
   }
}