package com.brownsoft.ag.individuo;

import com.brownsoft.ag.*;
import java.util.*;

/**
 * Title:        Motor AG
 * Description:
 * Copyright:    Copyright (c) 2003
 * Company:
 * @author Gustavo Brown
 * @version 1.0
 */

/** Esta clase se encarga de evaluar una expresion dado una instanca de IIndividuo o un IndividuoMultiple
 *
 *  El formato de la expresion permite constantes numericas, PI, operadores +, -, *, / y las siguientes
 *  funciones matematicas (pow, sqrt, cos, sin, tan, acos, asin, atan, floor, round, exp, ln, abs, int, frac, max, min)
 *  Las funciones max, min, y pow reciben 2 argumentos
 *  La funcion rnd() no recibe argumentos y retorna un nmero aleatrio entre 0 y 1
 *  Las demas funciones reciben 1 argumento
 *  Para acceder al valor funcionar de un Individuo, se realiza mediante x si se trata de una funcion de
 *  una sola variable, o de f(1)...f(n) en el caso de funciones de n variables.
 *  Por ejemplo, si se trata de un individuo simple (funcion de una sola variable):
 *    x * pow(2, x)
 *  Si se trata de una funcion de 3 variables:
 *    f(1) * pow(f(2), f(3))
 *
 * Mas ejemplos:
 *   x * int(x/abs(2*cos(x))-5)
 *   5*PI*cos(f(1) - sin(f(2))
 *
 * @see FuncionExpresion
 * @see IndividuoMultipleFuncionExpresion
 */
public class EvaluaExpresion
{
   String expresion;
   protected boolean isSimple;
   protected IIndividuo contained;

   protected EvaluaExpresion(String expresion) throws IllegalArgumentException
   {
      this.expresion = expresion;
      if (!matchParentesis(expresion))
      {
         throw new IllegalArgumentException("La expresion '" + expresion + "' no tiene los parntesis balanceados");
      }
   }

   /** Obtiene el fitness de este individuo
    */
   public double getFitness(IIndividuo individuo)
   { // Evalo la funcion
      try
      {
         this.contained = individuo;
         isSimple = individuo instanceof FuncionPuntoFijo;
         return eval(expresion);
      }
      catch (Exception e)
      {
         throw new MotorError("No se pudo evaluar la expresion '" + expresion + "': " + e.getMessage(), e);
      }
   }

   /** Evalua una expresion
    *  @param expr Expresion a evaluar
    *  @return double con el resultado
    */
   public double eval(String expr)
   {
      StringTokenizer tokenizer = new StringTokenizer(expr, "+-/*", true);
      return evaluate(tokenizer);
   }

   private double evaluate(StringTokenizer tokenizer)
   {
      double retVal = eval(tokenizer);
      double termino = 0;
      while (tokenizer.hasMoreTokens())
      {
         char operador = tokenizer.nextToken().charAt(0);
         // Obtengo el proximo termino
         switch (operador)
         {
            case '+':
               termino = evaluate(tokenizer);
               retVal = retVal + termino;
               break;
            case '-':
               termino = evaluate(tokenizer);
               retVal = retVal - termino;
               break;
            case '*':
               termino = eval(tokenizer);
               retVal = retVal * termino;
               break;
            case '/':
               termino = eval(tokenizer);
               retVal = retVal / termino;
               break;
         }
      }
      return retVal;
   }

   private double eval(StringTokenizer tokenizer)
   {
      String token = getNextToken(tokenizer);

      if (token.equalsIgnoreCase("-"))
      { // Si se trata de una expresion del tipo '-expr'
         return -eval(tokenizer);
      }
      if (token.equalsIgnoreCase("+"))
      { // Si se trata de una expresion del tipo '+expr'
         return eval(tokenizer);
      }

      // Hasta aqui tengo el termino a procesar
      // Entonces, veo si lo que tengo es una expresion entre parentesis,
      // un numero, o una funcion
      if (token.startsWith("("))
      {
         // Si es una expresion entre parentesis
         return eval(token.substring(1, token.length() - 1));
      }
      if (Character.isDigit(token.charAt(0)) || token.charAt(0) == '.')
      { // Si se trata de una constante (numero)
         return new Double(token).doubleValue();
      }
      if (token.equalsIgnoreCase("PI"))
      {
         return Math.PI;
      }
      if (token.equalsIgnoreCase("x"))
      { // Si se trata de la variable x
         if (isSimple)
         {
            return ( (FuncionPuntoFijo) contained).value();
         }
         else
         {
            throw new IllegalArgumentException("'x' solo puede ser usado en expresiones con individuos simples (1 sola variable)");
         }
      }
      if (token.startsWith("f("))
      { // Si se trata de alguna de las variables de un IndividuoMultiple
         if (!isSimple)
         {
            int idx = new Integer(token.substring(2, token.length() - 1)).intValue() - 1;
            try
            {
               return ( (FuncionPuntoFijo) ( (IndividuoMultiple) contained).getParameter(idx)).value();
            }
            catch (ArrayIndexOutOfBoundsException e)
            {
               throw new IllegalArgumentException("Indice erroneo en llamada a '" + token + "'. Rango: 1..." + ( (IndividuoMultiple) contained).parametros.length);
            }
         }
         else
         {
            throw new IllegalArgumentException(token + " solo puede ser usado en expresiones con individuos complejos (mas de 1 variable)");
         }
      }

      // Si no es nada de esto, debe ser una funcion
      String funcName = token.substring(0, token.indexOf('('));
      return evalFuncCall(funcName, token.substring(funcName.length() + 1, token.length() - 1));
   }

   /** Evalua la llamada a una funcion
    *  @param funcName Nombre de la funcion
    *  @param expr Expresion con los parametros
    *  @return resultado de la evaluacion
    */
   private double evalFuncCall(String funcName, String expr)
   {
      if (funcName.equalsIgnoreCase("rnd"))
      { // Si se trata de la funcion random
         return Math.random();
      }

      // Primero veo si es una funcion de 1 sola variable
      if (funcName.equalsIgnoreCase("abs"))
      {
         return Math.abs(eval(expr));
      }
      if (funcName.equalsIgnoreCase("int"))
      {
         return (double) ( (long) eval(expr));
      }
      if (funcName.equalsIgnoreCase("frac"))
      {
         double value = eval(expr);
         return value - (double) ( (long) value);
      }
      if (funcName.equalsIgnoreCase("sin"))
      {
         return Math.sin(eval(expr));
      }
      if (funcName.equalsIgnoreCase("asin"))
      {
         return Math.asin(eval(expr));
      }
      if (funcName.equalsIgnoreCase("cos"))
      {
         return Math.cos(eval(expr));
      }
      if (funcName.equalsIgnoreCase("acos"))
      {
         return Math.acos(eval(expr));
      }
      if (funcName.equalsIgnoreCase("tan"))
      {
         return Math.tan(eval(expr));
      }
      if (funcName.equalsIgnoreCase("atan"))
      {
         return Math.atan(eval(expr));
      }
      if (funcName.equalsIgnoreCase("floor"))
      {
         return Math.floor(eval(expr));
      }
      if (funcName.equalsIgnoreCase("round"))
      {
         return Math.round(eval(expr));
      }
      if (funcName.equalsIgnoreCase("ln") ||
          funcName.equalsIgnoreCase("log"))
      {
         double val = eval(expr);
         if (val <= 0)
         {
            throw new IllegalArgumentException("Argumento invalido (" + val + ") en llamada a log(" + expr + ")");
         }
         return Math.log(val);
      }
      if (funcName.equalsIgnoreCase("exp"))
      {
         return Math.exp(eval(expr));
      }
      if (funcName.equalsIgnoreCase("sqrt"))
      {
         return Math.sqrt(eval(expr));
      }

      // Si llego aca, se que es una funcion de 2 variables

      StringTokenizer paramTokenizer = new StringTokenizer(expr, ",", false);
      String sarg1, sarg2;
      try
      {
         sarg1 = getNextToken(paramTokenizer);
         sarg2 = getNextToken(paramTokenizer);
      }
      catch (NoSuchElementException e)
      {
         throw new IllegalArgumentException("No se puede procesar la funcion de 2 argumentos '" + funcName + "' en la expresion '" + expr + "'");
      }

      double arg1 = eval(sarg1);
      double arg2 = eval(sarg2);

      if (funcName.equalsIgnoreCase("pow"))
      {
         return Math.pow(arg1, arg2);
      }
      if (funcName.equalsIgnoreCase("max"))
      {
         return Math.max(arg1, arg2);
      }
      if (funcName.equalsIgnoreCase("min"))
      {
         return Math.min(arg1, arg2);
      }

      throw new IllegalArgumentException("No se puede parsear la funcion '" + funcName + "' en la expresion" + this.expresion);
   }

   private String getNextToken(StringTokenizer tokenizer)
   {
      String token = "";
      do
      {
         token += tokenizer.nextToken();
      }
      while (!matchParentesis(token));
      return token.trim();
   }

   private boolean matchParentesis(String token)
   {
      int cantLeft = 0, cantRight = 0;
      for (int i = 0; i < token.length(); i++)
      {
         char c = token.charAt(i);
         if (c == '(')
         {
            cantLeft++;
         }
         if (c == ')')
         {
            cantRight++;
         }
      }
      return cantLeft == cantRight;
   }

   public String toString()
   {
      return contained.toString();
   }
}
