package com.brownsoft.ag.individuo;

/** Esta clase implementa un individuo del tipo funcion discreta
 * de una sola variable (toma solo valores enteros).
 * Esta clase es abstracta. Las subclases de ella deben implementar el
 * mtodo getFitness() obteniendo el valor (de tipo double) del metodo value()
 * @author Gustavo Brown
 * @version 1.0
 */
public abstract class FuncionDiscreta extends IndividuoBitStreamSimple implements IFuncion
{
   protected long from;
   protected long to;
   protected long width;
   protected String varName;

   /** Constructor
    * Los distintos puntos del espacio de bsqueda estan igualmente distanciados
    * @param from limite izquierdo del dominio de la funcion (inclusive)
    * @param to limite derecho del dominio de la funcion (inclusive)
    * @note: Este constructor considera x como el nombre de la variable
    */
   public FuncionDiscreta(long from, long to)
   {
      this(from, to, "x");
   }

   /** Constructor
    * Los distintos puntos del espacio de bsqueda estan igualmente distanciados
    * @param from limite izquierdo del dominio de la funcion (inclusive)
    * @param to limite derecho del dominio de la funcion (inclusive)
    * @param varName nombre de la variable de la funcin
    */
   public FuncionDiscreta(long from, long to, String varName)
   {
      // La cantidad de bits minima necesaria para poder representar todos los varlores
      // entre from y to la calculo como (Log2(to - from - 1) + 1)
      super( (int) (Math.log(to - from - 1) / Math.log(2)) + 1);
      this.from = from;
      this.to = to;
      width = to - from;
      this.varName = varName;
   }

   public String toString()
   {
      return varName + " = " + value();
   }

   /** Obtiene el valor que toma la variable independiente
    * @return Retorna el valor que toma la variable independiente para este individuo
    */
   public double value()
   {
      return ( (long) ( (double)super.value / (super.maxValue - 1) * (double) width)) + from;
   }

   /** Obtiene el fenotipo de este individuo
    * @return Double con el valor que toma la variable independenite para este individuo
    */
   public Object getFenotipo()
   {
      return new Double(value());
   }

   /** Las subclases deben implementar este mtodo
    */
   public abstract double getFitness();
}
