package com.brownsoft.ag.individuo;

import com.brownsoft.ag.*;

/** Esta clase es utilizada para crear individuos multiparmetros
 * Cada uno de los parametros a su vez es una instancia de Individuo
 * La clase es abstracta, pues se deja a las subclases el trabajo de definir
 * el fenotipo y el fitness.
 * Con la funcion getParameter(int index) se obtiene el parametro numero 'index'
 * @see OperadorMultiple
 * @author Gustavo Brown
 * @version 1.0
 */
public abstract class IndividuoMultiple implements IIndividuo
{
   protected IIndividuo[] parametros;
   protected IIndividuo[] parents;

   /** Constructor
    * @param parametros Array de IIndividuo con los parametros de este individuo
    */
   public IndividuoMultiple(IIndividuo[] parametros)
   {
      this.parametros = parametros;
   }

   /** Crea una copia con el mismo material gentico que este individuo
    * @return Nuevo individuo
    */
   public IIndividuo getCopy()
   {
      try
      {
         IndividuoMultiple individuo = (IndividuoMultiple)this.clone();
         individuo.parents = new IIndividuo[0];
         individuo.parametros = new IIndividuo[parametros.length];
         for (int i = 0; i < parametros.length; i++)
         {
            individuo.setParameter(i, parametros[i].getCopy());
         }
         return individuo;
      }
      catch (CloneNotSupportedException e)
      {
         throw new MotorError("No se pudo clonar el individuo: " + toString(), e);
      }
   }

   /** Setea el material gentico de este individuo al azar
    */
   public void setRandom()
   {
      for (int i = 0; i < parametros.length; i++)
      {
         parametros[i].setRandom();
      }
   }

   /** Obtiene el material genetico de este individuo
    * @return Object con el material genetico del individuo
    */
   public Object getValue()
   {
      throw new MotorError("Las instancias de IndividuoMultiple NO tienen definido el metodo getValue");
   }

   /** Setea el valor de este individuo
    * @param value Valor de este individuo
    */
   public void setValue(Object value)
   {
      throw new MotorError("Las instancias de IndividuoMultiple NO tienen definido el metodo setValue");
   }

   /** Obtiene el fitness de este individuo
    * Este mtodo es abstracto. Las subclases deben definirlo
    */
   public abstract double getFitness();

   /** Obtiene el fenotipo de este individuo
    * Este mtodo es abstracto. Las subclases deben definirlo
    */
   public abstract Object getFenotipo();

   /** Setea los padres de este individuo
    * @param parents padres de este individuo
    */
   public void setParents(IIndividuo[] parents)
   {
      this.parents = parents;
   }

   /** Obtiene los padres de este individuo
    * @return padres de este individuo
    */
   public IIndividuo[] getParents()
   {
      return parents;
   }

   /** Obtiene el parametro asociado a un ndice
    * @param index indice del parametro a obtener
    * @return IIndividuo asociado a dicho parmetro
    */
   public IIndividuo getParameter(int index) throws MotorError
   {
      try
      {
         return parametros[index];
      }
      catch (IndexOutOfBoundsException e)
      {
         throw new MotorError("Error en getParameter. El numero de parametro no debe exceder " + (parametros.length - 1) + "(index=" + index + ")", e);
      }
   }

   /** Obtiene el numero de parametro asociado a un parametro
    * @param IIndividuo asociado a algun parmetro
    * @return index indice asociado a dicho parametro
    */
   public int getParameterIndex(IIndividuo parameter) throws MotorError
   {
      for (int i = 0; i < parametros.length; i++)
      {
         if (parametros[i].equals(parameter))
         {
            return i;
         }
      }
      throw new MotorError("Error en getParameterIndex. El individuo '" + parameter + "' no es parte de este individuo.");
   }

   /** Coloca el individuo asociado al parmetro i-simo.
    * Es utilizado internamente por los operadores que interactual con IndividuoMultiple
    * para construir nuevas instancias de IndividuoMultiple
    * @param index indice del parametro a setear
    * @param parameter nuevo IIndividuo asociado al parmetro i-simo
    */
   public void setParameter(int index, IIndividuo parameter)
   {
      parametros[index] = parameter;
   }

   public String toString()
   {
      String toString = parametros[0].toString();
      for (int i = 1; i < parametros.length; i++)
      {
         toString += "; " + parametros[i].toString();
      }
      return toString;
   }

   /** Hashcode para este individuo
    *  @return int con el Hashcode del individuo
    */
   public int hashCode()
   {
      int hashCode = 0;
      for (int i = 0; i < parametros.length; i++)
      {
         hashCode ^= parametros[i].hashCode();
      }
      return hashCode;
   }

   /** Sobrecarga del metodo equals
    *  @param Object a comparar
    *  @return boolean indicando si son iguales
    */
   public boolean equals(Object o)
   {
      try
      {
         IndividuoMultiple obj = (IndividuoMultiple) o;
         for (int i = 0; i < parametros.length; i++)
         {
            if (!getParameter(i).equals(obj.getParameter(i)))
            {
               return false;
            }
         }
         return true;
      }
      catch (Exception e)
      {
         return false;
      }
   }

}
