package com.brownsoft.ag.seleccion;

import com.brownsoft.ag.*;
import java.util.*;

/** Esta clase implementa el operador de seleccion del tipo 'Muestreo Estocastico Universal'
 *  (Stochastic Universal Selection', que es un metodo de seleccion analogo al de la ruleta
 *  pero con M punteros equidistantes (donde M es la cantidad de individuos a seleccionar).
 *  Es decir que se realizan TODAS las selecciones asociadas a una iteracin de a una vez.
 *  Este metodo no esta sesgado y su dispersion es la minima posible.
 *  El tamao de cada posicion depende del fitness relativo que tenga cada individuo
 *  respecto a los dems. Se debe tener en cuenta que si una posicion es ms grande que el espaciado
 *  entre 2 punteros de la ruleta, ese individuo va a ser seleccionado al menos 1 vez
 * @author Gustavo Brown
 * @version 1.0
 */
public class SeleccionEstocasticaUniversal extends SeleccionRuleta
{
   private IIndividuo[] seleccionados;
   private int index;

   /** Avisa que se inicia una nueva iteracin
    */
   public void inicioIteracion()
   {
      // Primero calculo los rouletePoints, etc
      super.inicioIteracion();
      int popSize = super.getPopSize();

      // Ahora selecciono todos los individuos
      double distancia = 1.0 / popSize;
      double curPointer = PRNG.nextProbability();
      IIndividuo[] temporal = new IIndividuo[popSize];
      seleccionados = new IIndividuo[popSize];
      for (int i = 0; i < popSize; i++)
      { // Voy seleccionando los individuos
         temporal[i] = rouletteSelect(curPointer);
         curPointer += distancia;
         curPointer %= 1;
      }

      // Ok, ahora tengo todos los individuos seleccionados, pero estan
      // ordenados por fitness, asi que hacemos un shuffle para desordenarlos
      for (int i = 0; i < popSize; i++)
      {
         // Obtengo la siguiente posicion al azar
         // y me aseguro que no sea una que ya haya elegido
         int shuffled = PRNG.nextInt(popSize);
         while (temporal[shuffled] == null)
         {
            shuffled++;
            shuffled %= popSize;
         }
         seleccionados[i] = temporal[shuffled];
         temporal[shuffled] = null;
      }
      index = 0;
   }

   /** Selecciona individuos
    * @return IIndividuo[] con los individuos seleccionados
    */
   public IIndividuo[] seleccionar()
   {
      // Obtengo los proximos 2 individuos
      IIndividuo[] seleccionados = new IIndividuo[2];
      seleccionados[0] = this.seleccionados[index++];
      seleccionados[1] = this.seleccionados[index++];
      return seleccionados;
   }

   /** Selecciona un individuo mediante una ruleta
    */
   private IIndividuo rouletteSelect(double value)
   {
      int i;
      for (i = 0; i < roulettePoints.length; i++)
      {
         if (roulettePoints[i] >= value)
         {
            return (IIndividuo) poblacionIteracion.elementAt(i);
         }
      }
      throw new MotorError("SeleccionEstocasticaUniversal.rouleteSelect fallo: value= " + value);
   }
}
