package com.brownsoft.ag.seleccion;

import com.brownsoft.ag.*;
import java.util.*;

/** Esta clase en cada seleccion ejecuta N selecciones de la
 * Seleccion indicada en el constructor
 * @author Gustavo Brown
 * @version 1.0
 */
public class SeleccionNSelecciones implements ISeleccion
{
   protected ISeleccion seleccion;
   protected int N;

   /** Constructor
    * @param seleccion ISeleccion a ejecutar en cada seleccion
    * @param N cantidad de veces a ejecutar la seleccion por cada iteracion
    */
   public SeleccionNSelecciones(ISeleccion seleccion, int N) throws MotorException
   {
      if (N < 1)
      {
         throw new MotorException("La cantidad de veces a ejecutar la seleccion por iteracion debe ser mayor que 0 (N=" + N + ")");
      }
      this.seleccion = seleccion;
      this.N = N;
   }

   /** Inicializa el operador de seleccion
    * @param motor MotorAG asociado
    */
   public void inicializar(MotorAG motor)
   {
      seleccion.inicializar(motor);
   }

   /** Avisa que se inicia una nueva iteracin
    */
   public void inicioIteracion()
   {
      seleccion.inicioIteracion();
   }

   /** Indica que se finaliza la iteracion
    */
   public void finIteracion()
   {
      seleccion.finIteracion();
   }

   /** Avisa que finaliz la ejecucin del algoritmo
    */
   public void finAlgoritmo()
   {
      seleccion.finAlgoritmo();
   }

   /** Selecciona individuos
    * @return IIndividuo[] con los individuos seleccionados
    */
   public IIndividuo[] seleccionar()
   {
      IIndividuo[] seleccionados = new IIndividuo[0];

      // Ejecuto N veces el operador de seleccion subyacente y junto todos los
      // individuos seleccionados
      for (int i = 0; i < N; i++)
      {
         IIndividuo[] seleccionadosIteracion = seleccion.seleccionar();
         IIndividuo[] temp = new IIndividuo[seleccionados.length + seleccionadosIteracion.length];
         System.arraycopy(seleccionados, 0, temp, 0, seleccionados.length);
         System.arraycopy(seleccionadosIteracion, 0, temp, seleccionados.length, seleccionadosIteracion.length);
         seleccionados = temp;
      }

      return seleccionados;
   }

   /** Pone individuos en el pool intermedio
    * @param individuos individuos a ingresar en el pool intermedio
    */
   public void ponerEnPoolIntermedio(IIndividuo[] individuos)
   {
      seleccion.ponerEnPoolIntermedio(individuos);
   }

   /** Genera una nueva poblacion a partir de un pool
    */
   public Vector generarNuevaPoblacion()
   {
      return seleccion.generarNuevaPoblacion();
   }
}
