package com.brownsoft.ag.seleccion;

import com.brownsoft.ag.*;
import java.util.*;

/** Esta clase implementa el operador de seleccion del tipo Ruleta, es decir
 *  que selecciona de a 2 individuos lanzando 2 veces una ruleta de n posiciones.
 *  El tamao de cada posicion depende del fitness relativo que tenga cada individuo
 *  respecto a los dems
 * @author Gustavo Brown
 * @version 1.0
 */
public class SeleccionRuleta implements ISeleccion
{
   private MotorAG motor;
   protected Vector poblacionIteracion;
   private Vector poolIntermedio;
   protected double[] roulettePoints;

   /** Inicializa el operador de seleccion
    * @param motor MotorAG asociado
    */
   public void inicializar(MotorAG motor)
   {
      this.motor = motor;
      poolIntermedio = new Vector();
   }

   /** Avisa que se inicia una nueva iteracin
    */
   public void inicioIteracion()
   {
      poolIntermedio.removeAllElements();
      double sumFitness = 0;
      // En el mtodo de la ruleta lo primero que hacemos es obtener la proporcion de la ruleta
      // de cada individuo
      poblacionIteracion = (Vector) motor.getPoblacion().clone();
      int popSize = getPopSize();
      roulettePoints = new double[popSize];
      int i = 0;
      for (Enumeration enum = poblacionIteracion.elements(); enum.hasMoreElements(); )
      {
         IIndividuo individuo = (IIndividuo) enum.nextElement();
         double fitness = motor.getFitness(individuo);
         if (fitness < 0)
         {
            // Para este operador de seleccion, el fitness NO puede ser negativo
            // Utilizar un fitnessScaler (por ejemplo el FitnessScalerPositivo) para
            // tener todos los fitness de la poblacion positivos
            // @see MotorAG.setFitnessScaler
            throw new MotorError("El fitness no puede ser negativo en operador de seleccion SeleccionRuleta (individuo: " + individuo.toString() + ")");
         }
         sumFitness += fitness;
         roulettePoints[i++] = fitness;
      }

      // Ahora calculo la ubicacion en la ruleta de cada individuo
      double lastFitness = 0;
      for (i = 0; i < popSize; i++)
      {
         roulettePoints[i] = roulettePoints[i] / sumFitness + lastFitness;
         lastFitness = roulettePoints[i];
      }

      // Por errores de redondeo, al ltimo le asigno 1 para tener el total de la ruleta
      roulettePoints[popSize - 1] = 1;
   }

   /** Selecciona individuos
    * @return IIndividuo[] con los individuos seleccionados
    */
   public IIndividuo[] seleccionar()
   {
      // Tiro la ruleta 2 veces y selecciono 2 individuos
      IIndividuo[] seleccionados = new IIndividuo[2];
      seleccionados[0] = rouletteSelect();
      seleccionados[1] = rouletteSelect();
      return seleccionados;
   }

   /** Selecciona un individuo mediante una ruleta
    */
   private IIndividuo rouletteSelect()
   {
      double value = PRNG.nextProbability();
      int i;
      for (i = 0; i < roulettePoints.length; i++)
      {
         if (roulettePoints[i] >= value)
         {
            return (IIndividuo) poblacionIteracion.elementAt(i);
         }
      }
      throw new MotorError("SeleccionRuleta.rouleteSelect fallo: value= " + value);
   }

   /** Pone individuos en el pool intermedio
    * @param individuos individuos a ingresar en el pool intermedio
    */
   public void ponerEnPoolIntermedio(IIndividuo[] individuos)
   {
      for (int i = 0; i < individuos.length; i++)
      {
         poolIntermedio.addElement(individuos[i]);
      }
   }

   protected int getPopSize()
   {
      return poblacionIteracion.size();
   }

   /** Genera una nueva poblacion a partir de un pool
    */
   public Vector generarNuevaPoblacion()
   {
      return (Vector) poolIntermedio.clone();
   }

   /** Indica que se finaliza la iteracion
    */
   public void finIteracion()
   {
      ;
   }

   /** Avisa que finaliz la ejecucin del algoritmo
    */
   public void finAlgoritmo()
   {
      ;
   }
}
