package com.brownsoft.ag.cruzamiento;

import com.brownsoft.ag.*;
import com.brownsoft.ag.individuo.*;
import java.io.*;

/** Esta clase implementa cruzamiento en N puntos
 * @author Gustavo Brown
 * @version 1.0
 */
public class CruzamientoMPX implements ICruzamiento
{
   private int cantCrosses;

   /** Constructor
    * @param cantCrosses Cantidad de puntos de corte
    */
   public CruzamientoMPX(int cantCrosses) throws InvalidCrossException
   {
      this.cantCrosses = cantCrosses;
   }

   /** Inicializa el operador de cruzamiento
    * @param motor MotorAG asociado
    */
   public void inicializar(MotorAG motor)
   {
      ;
   }

   /** Avisa que se inicia una nueva iteracin
    */
   public void inicioIteracion()
   {
      ;
   }

   /** Avisa que finalizo la iteracin
    */
   public void finIteracion()
   {
      ;
   }

   /** Avisa que finaliz la ejecucin del algoritmo
    */
   public void finAlgoritmo()
   {
      ;
   }

   /** Cruza un grupo de individuos
    * @param seleccioandos grupo de individuos a cruzar
    * @return grupo de individuos cruzados
    * @exception InvalidCrossException en caso de que ocurre
    */
   public IIndividuo[] cruzar(IIndividuo[] seleccionados) throws InvalidCrossException
   {
      if (seleccionados.length != 2)
      {
         throw new InvalidCrossException("CruzamientoMPX: La cantidad de individuos a cruzar debe ser 2");
      }
      IndividuoBitStreamSimple padreX, padreY;

      try
      {
         // Primero obtengo a los 2 individuos
         padreX = (IndividuoBitStreamSimple) seleccionados[0];
         padreY = (IndividuoBitStreamSimple) seleccionados[1];
      }
      catch (ClassCastException invalidCast)
      {
         throw new InvalidCrossException("CruzamientoMPX: Los individuos no son del tipo 'IndividuoBitStreamSimple'", invalidCast);
      }

      // Ahora obtengo los puntos de corte
      int bitWidth = padreX.getBitWidth();
      if (cantCrosses >= bitWidth)
      {
         throw new InvalidCrossException("CruzamientoMPX: Cantidad de puntos de corte(" + cantCrosses + ") mayor que el largo del bitstream de los indiviudos(" + bitWidth + ")");
      }

      int[] crossPoints = new int[cantCrosses];
      for (int i = 0; i < cantCrosses; i++)
      {
         crossPoints[i] = PRNG.nextInt(bitWidth);
         for (int j = 0; j < i; j++)
         {
            if (crossPoints[j] == crossPoints[i])
            { // Si ese punto de corte ya exista, reelegimos uno nuevo
               i--;
               break;
            }
         }
      }

      // Ahora realizo el cruzamiento
      // Para ello lo que voy a crear son 2 mscaras (mask1 y mask2)
      // Luego le aplico estas mascaras a los padres para obtener los 2 hijos
      long mask1 = 0, mask2 = 0;
      long curValue = 0;
      for (int i = 0; i < cantCrosses; i++)
      {
         int minPoint = Integer.MAX_VALUE;
         int minPos = -1;
         for (int j = 0; j < cantCrosses; j++)
         {
            if (crossPoints[j] < minPoint)
            {
               minPos = j;
               minPoint = crossPoints[j];
            }
         }
         crossPoints[minPos] = Integer.MAX_VALUE;
         long thisValue = (long) Math.pow(2, minPoint + 1) - 1;
         long tempValue = mask2;
         mask2 = mask1 + thisValue - curValue;
         mask1 = tempValue;
         curValue = thisValue;
      }

      mask1 += ( (long) Math.pow(2, bitWidth) - 1) - curValue;

      // Ahora si realizo el cruzamiento
      long valueX = ( (Long) padreX.getValue()).longValue();
      long valueY = ( (Long) padreY.getValue()).longValue();

      long hijoX = (valueX & mask1) | (valueY & mask2);
      long hijoY = (valueX & mask2) | (valueY & mask1);

      IIndividuo[] hijos = new IIndividuo[]
          {padreX.getCopy(), padreY.getCopy()};
      hijos[0].setValue(new Long(hijoX));
      hijos[1].setValue(new Long(hijoY));
      hijos[0].setParents(new IIndividuo[]
                          {padreX, padreY});
      hijos[1].setParents(new IIndividuo[]
                          {padreX, padreY});

      return hijos;
   }
}