package com.brownsoft.ag.fitnessScaler;

import com.brownsoft.ag.*;

/** Esta clase debe ser extendida por las clases que desean
 *  escalar el fitness de un Individuo (por ejemplo para la seleccion)
 *  El fitness escalado de un individuo se obtiene mediante la ejecucin
 *  de MotorAG.getFitness(IIndividuo)
 *  @see MotorAG.getFitness
 * @author Gustavo Brown
 * @version 1.0
 */

public abstract class FitnessScaler implements IFitnessScaler
{
   private IFitnessScaler parent;

   /** Crea una instancia de FitnessScaler
    *  @param parent el IFitnessScaler sobre el cual actuar
    *  @nota El ultimo IFitnessScaler debe ser FitnessScalerSink
    */
   protected FitnessScaler(IFitnessScaler parent)
   {
      this.parent = parent;
   }

   protected MotorAG motor;

   /** Setea el motor asociado a este FitnessScaler
    *  Este metodo es ejecutado por MotorAG
    *  @param motor el MotorAG
    */
   public final void initScaler(MotorAG motor)
   {
      parent.initScaler(motor);
      this.motor = motor;
   }

   /** Obtiene la instancia de MotorAG asociada a este scaler
    *  @return MotorAG la instancia de MotorAG asociada a este scaler
    */
   public MotorAG getMotor()
   {
      return motor;
   }

   private double originalFitness;

   /** Escala un fitness
    *  @param individuo individuo a obtener su fitness
    *  @return fitness escalado
    */
   public final double getScaledFitness(IIndividuo individuo)
   {
      return scaleFitness(parent.getScaledFitness(individuo));
   }

   /** Escala un fitness
    *  @param fitness Fitness a escalar
    *  @return fitness escalado
    */
   public final double getScaledFitness(double fitness)
   {
      return scaleFitness(parent.getScaledFitness(fitness));
   }

   /** Obtiene el IIndividuo que esta siendo escalado en este momento
    **/
   public final IIndividuo getProcessingIndividual()
   {
      return parent.getProcessingIndividual();
   }

   /** Le avisa al Scaler y a su parent que se inicia la iteracion
    *  es decir, debe llamara parent.fireInicioIteracion() y luego a inicioIteracion
    */
   public void fireInicioIteracion()
   {
      parent.fireInicioIteracion();
      inicioIteracion();
   }

   /** Avisa que se inicia una nueva iteracin
    */
   public void inicioIteracion()
   {
      ;
   }

   /** Este metodo debe ser definido por las subclases de FitnessScaler
    *  @nota las subclases deben llamar a scaleFitness para obtener el fitness a escalar
    *  @return fitness escalado
    */
   protected double scaleFitness(double fitness)
   {
      // La implementacin base de scaleFitness retorna el fitness original para el individuo
      return originalFitness;
   }
}