package com.brownsoft.ag.individuo;

/** Esta clase implementa un individuo del tipo funcion de punto fijo
 * de una sola variable.
 * Esta clase es abstracta. Las subclases de ella deben implementar el
 * mtodo getFitness() obteniendo el valor (de tipo double) del metodo value()
 * @author Gustavo Brown
 * @version 1.0
 */
public abstract class FuncionPuntoFijo extends IndividuoBitStreamSimple implements IFuncion
{
   protected double from;
   protected double to;
   protected double width;
   protected String varName;
   protected boolean isDiscrete;

   /** Constructor
    * @param from limite izquierdo del dominio de la funcion (inclusive)
    * @param to limite derecho del dominio de la funcion (inclusive)
    * @param bitWidth cantidad de bits en la representacion del individuo. Los
    * distintos puntos del espacio de bsqueda estan igualmente distanciados. Si
    * el bitWidth es 0, se toman solamente valores enteros para la variable
    * independiente (o sea, trabaja igual que FuncionDiscreta)
    * @note: Este constructor considera x como el nombre de la variable
    */
   public FuncionPuntoFijo(double from, double to, int bitWidth)
   {
      this(from, to, bitWidth, "x");
   }

   /** Constructor
    * @param from limite izquierdo del dominio de la funcion (inclusive)
    * @param to limite derecho del dominio de la funcion (inclusive)
    * @param bitWidth cantidad de bits en la representacion del individuo. Los
    * distintos puntos del espacio de bsqueda estan igualmente distanciados. Si
    * el bitWidth es 0, se toman solamente valores enteros para la variable
    * independiente (o sea, trabaja igual que FuncionDiscreta)
    * @param varName nombre de la variable de la funcin
    */
   public FuncionPuntoFijo(double from, double to, int bitWidth, String varName)
   {
      super(bitWidth != 0 ? bitWidth : (int) (Math.log(to - from - 1) / Math.log(2)) + 1);
      this.from = from;
      this.to = to;
      width = to - from;
      this.varName = varName;
      isDiscrete = bitWidth == 0; // Indica si la funcion toma solo valores enteros
   }

   public String toString()
   {
      return varName + " = " + value();
   }

   /** Obtiene el valor que toma la variable independiente
    * @return Retorna el valor que toma la variable independiente para este individuo
    */
   public double value()
   {
      if (isDiscrete)
      {
         return Math.floor( ( (double)super.value / (super.maxValue - 1) * (double) width) + from);
      }
      else
      {
         return ( (double)super.value / (super.maxValue - 1) * (double) width) + from;
      }
   }

   /** Obtiene el fenotipo de este individuo
    * @return Double con el valor que toma la variable independenite para este individuo
    */
   public Object getFenotipo()
   {
      return new Double(value());
   }

   /** Las subclases deben implementar este mtodo
    */
   public abstract double getFitness();
}
