package com.brownsoft.ag.individuo;

import com.brownsoft.ag.*;
import java.util.*;

/** Esta clase implementa un Individuo del tipo bitstream de tamao <= 63bits
 * @author Gustavo Brown
 * @version 1.0
 */
public abstract class IndividuoBitStreamSimple implements IIndividuo
{
   protected int bitWidth;
   protected long maxValue;
   protected long value;
   protected IIndividuo[] parents;

   /** Constructor de IndividuoBitStreamSimple
    *  @param largoBitStream cantidad de bits a utilizar para representar este individuo (de 1 a 63)
    */
   public IndividuoBitStreamSimple(int largoBitStream)
   {
      if (largoBitStream > 63 || largoBitStream < 1)
      {
         throw new MotorError("La cantidad de bits para representacin de los individuos de tipo IndividuoBitStreamSimple debe ser entre 1 y 63 [usado: " + largoBitStream + "]");
      }
      this.bitWidth = largoBitStream;
      this.maxValue = ( (long) 1) << largoBitStream;
      this.parents = new IIndividuo[0];
      setRandom();
   }

   /** Crea una copia con el mismo material gentico que este individuo
    * @return Nuevo individuo
    */
   public IIndividuo getCopy()
   {
      try
      {
         IndividuoBitStreamSimple individuo = (IndividuoBitStreamSimple)this.clone();
         individuo.setValue(this.getValue());
         individuo.parents = new IIndividuo[this.parents.length];
         System.arraycopy(this.parents, 0, individuo.parents, 0, this.parents.length);
         return individuo;
      }
      catch (CloneNotSupportedException e)
      {
         throw new MotorError("No se pudo clonar el individuo: " + toString(), e);
      }
   }

   /** Obtiene el material genetico de este individuo
    * @return Object con el material genetico del individuo
    */
   public Object getValue()
   {
      return new Long(value);
   }

   /** Setea el valor de este individuo
    * @param value Valor de este individuo
    */
   public void setValue(Object value)
   {
      this.value = ( (Long) value).longValue();
   }

   /** Setea el material gentico de este individuo al azar
    */
   public void setRandom()
   {
      value = PRNG.nextLong(maxValue);
   }

   /** Obtiene el largo del cromosoma
    * @return largo (en bits)
    */
   public int getBitWidth()
   {
      return bitWidth;
   }

   /** Setea los padres de este individuo
    * @param parents padres de este individuo
    */
   public void setParents(IIndividuo[] parents)
   {
      this.parents = parents;
   }

   /** Obtiene los padres de este individuo
    * @return padres de este individuo
    */
   public IIndividuo[] getParents()
   {
      return parents;
   }

   /** Hashcode para este individuo
    *  @return int con el Hashcode del individuo
    */
   public int hashCode()
   {
      return (int) value;
   }

   /** Sobrecarga del metodo equals
    *  @param Object a comparar
    *  @return boolean indicando si son iguales
    */
   public boolean equals(Object o)
   {
      try
      {
         return getValue().equals( ( (IndividuoBitStreamSimple) o).getValue());
      }
      catch (Exception e)
      {
         return false;
      }
   }
}
