package com.brownsoft.ag.seleccion;

import com.brownsoft.ag.*;
import java.util.*;

/** Esta clase aplica el modelo elitista sobre cualquier seleccion
 * Al generar la nueva poblacin, si el mejor individuo hasta el momento
 * no esta en la ella, se inserta como el elemento N+1
 * Seleccion indicada en el constructor
 * @author Gustavo Brown
 * @version 1.0
 */
public class SeleccionElitista implements ISeleccion
{
   protected ISeleccion seleccion;
   protected IIndividuo bestIndividual = null;
   protected MotorAG motor;

   /** Constructor
    * @param seleccion ISeleccion a ejecutar en cada seleccion
    */
   public SeleccionElitista(ISeleccion seleccion) throws MotorException
   {
      this.seleccion = seleccion;
   }

   /** Inicializa el operador de seleccion
    * @param motor MotorAG asociado
    */
   public void inicializar(MotorAG motor)
   {
      seleccion.inicializar(motor);
      this.motor = motor;
   }

   /** Avisa que se inicia una nueva iteracin
    */
   public void inicioIteracion()
   {
      // Aqui obteneo el mejor individuo hasta el momento
      bestIndividual = motor.getBestIndividuo();
      seleccion.inicioIteracion();
   }

   /** Indica que se finaliza la iteracion
    */
   public void finIteracion()
   {
      seleccion.finIteracion();
   }

   /** Avisa que finaliz la ejecucin del algoritmo
    */
   public void finAlgoritmo()
   {
      seleccion.finAlgoritmo();
   }

   /** Selecciona individuos
    * @return IIndividuo[] con los individuos seleccionados
    */
   public IIndividuo[] seleccionar()
   {
      return seleccion.seleccionar();
   }

   /** Pone individuos en el pool intermedio
    * @param individuos individuos a ingresar en el pool intermedio
    */
   public void ponerEnPoolIntermedio(IIndividuo[] individuos)
   {
      seleccion.ponerEnPoolIntermedio(individuos);
   }

   /** Genera una nueva poblacion a partir de un pool
    */
   public Vector generarNuevaPoblacion()
   {
      Vector nuevaPoblacion = seleccion.generarNuevaPoblacion();
      // Veo si el mejor individuo de la iteracion pasada esta en la nueva poblacion
      // y si no est lo agrego
      if (bestIndividual != null && !nuevaPoblacion.contains(bestIndividual))
      {
         nuevaPoblacion.addElement(bestIndividual);
      }
      return nuevaPoblacion;
   }
}
