package com.brownsoft.ag.seleccion;

import com.brownsoft.ag.*;
import java.util.*;

/** Esta clase implementa la seleccion por torneo de Q individuos
 * tomados de K individuos elegidos al azar de la poblacin.
 * @author Gustavo Brown
 * @version 1.0
 */
public class SeleccionTorneoQK implements ISeleccion
{
   private MotorAG motor;
   private Vector poblacionIteracion;
   private Vector poolIntermedio;
   private int Q, K;

   /** Constructor
    * @param Q cantidad de individuos a elegir en cada torneo
    * @param K cantidad de competidores de cada torneo
    */
   public SeleccionTorneoQK(int Q, int K) throws MotorException
   {
      if (Q > K)
      {
         throw new MotorException("El argumento Q no puede ser mayor que K en la Seleccion por Torneo Q-K (Q=" + Q + ",K=" + K + ")");
      }
      if (Q < 1 || K < 1)
      {
         throw new MotorException("Los argumentos Q y K deben ser igual o mayores que 1 (Q=" + Q + ",K=" + K + ")");
      }
      this.Q = Q;
      this.K = K;
   }

   /** Inicializa el operador de seleccion
    * @param motor MotorAG asociado
    */
   public void inicializar(MotorAG motor)
   {
      this.motor = motor;
      poolIntermedio = new Vector();
   }

   /** Avisa que se inicia una nueva iteracin
    */
   public void inicioIteracion()
   {
      poolIntermedio.removeAllElements();
   }

   /** Selecciona individuos
    * @return IIndividuo[] con los individuos seleccionados
    */
   public IIndividuo[] seleccionar()
   {
      IIndividuo[] torneo = new IIndividuo[K];
      IIndividuo[] seleccionados = new IIndividuo[Q];

      // Lo primero que tenemos que hacer es elegir K individuos al azar
      Vector poblacion = (Vector) motor.getPoblacion();
      int popSize = poblacion.size();
      for (int i = 0; i < K; i++)
      {
         torneo[i] = (IIndividuo) poblacion.elementAt(PRNG.nextInt(popSize));
      }

      // Ahora elegimos a los Q mejores mediante un bubble sort
      for (int i = 0; i < Q; i++)
      {
         for (int j = K - 1; j > i; j--)
         {
            if (motor.getFitness(torneo[j - 1]) < motor.getFitness(torneo[j]))
            {
               IIndividuo temp = torneo[j];
               torneo[j] = torneo[j - 1];
               torneo[j - 1] = temp;
            }
         }
         seleccionados[i] = torneo[i];
      }

      // Y retornamos los seleccionados
      return seleccionados;
   }

   /** Pone individuos en el pool intermedio
    * @param individuos individuos a ingresar en el pool intermedio
    */
   public void ponerEnPoolIntermedio(IIndividuo[] individuos)
   {
      for (int i = 0; i < individuos.length; i++)
      {
         poolIntermedio.addElement(individuos[i]);
      }
   }

   /** Genera una nueva poblacion a partir de un pool
    */
   public Vector generarNuevaPoblacion()
   {
      return (Vector) poolIntermedio.clone();
   }

   /** Indica que se finaliza la iteracion
    */
   public void finIteracion()
   {
      ;
   }

   /** Avisa que finaliz la ejecucin del algoritmo
    */
   public void finAlgoritmo()
   {
      ;
   }
}
