package com.brownsoft.gui;

import com.brownsoft.ag.*;

import java.awt.*;
import java.awt.event.*;
import java.util.*;
import com.brownsoft.ag.individuo.*;

/**
 * Title:        Motor AG
 * Description: Esta clase es la encargada de ir graficando una funcion
 * Copyright:    Copyright (c) 2003
 * Company:
 * @author Gustavo Brown
 * @version 1.0
 */

public class Graph extends Component implements ComponentListener, IMotorListener, MouseListener, MouseMotionListener
{
   private int screenWidth, screenHeight;
   private IMsgSink statusBar;
   private MotorAG motor;
   private Point mouseLocation = new Point(0, 0);
   private boolean running = false;

   public Graph(IMsgSink statusBar)
   {
      this.statusBar = statusBar;
      super.addComponentListener(this);
      super.addMouseListener(this);
      super.addMouseMotionListener(this);
   }

   IIndividuo bestIndividual = null;
   private boolean isDeJong = false;
   public void initialize(MotorAG motor)
   {
      this.motor = motor;
      running = true;
      motor.addListener(this);
      isDeJong = motor.getGeneracion()instanceof GeneracionSimpleDeJong;
   }

   /** Muestra un mensaje en la statusBar
    *  @param msg Mensaje a mostrar
    */
   public void showMsg(String msg)
   {
      statusBar.showMsg(msg);
   }

   public int getWidth()
   {
      return getSize().width;
   }

   public int getHeight()
   {
      return getSize().height;
   }

   static Font bigFont = new Font("Dialog", Font.BOLD, 14);
   static Font smallFont = new Font("Dialog", Font.PLAIN, 12);

   public void paint(Graphics g)
   {
      if (motor == null)
      {
         g.setColor(Color.green);
         g.fillRect(0, 0, getWidth(), getHeight());
         return;
      }
      if (!running)
      {
         g.setColor(Color.yellow);
         g.fillRect(0, 0, getWidth(), getHeight());
      }
      else
      {
         g.setColor(Color.lightGray);
         g.fillRect(0, 0, getWidth(), getHeight());
      }

      if (dragging)
      {
         g.setColor(Color.cyan);
         g.fillRect(Math.min(draggingInit.x, draggingEnd.x), Math.min(draggingInit.y, draggingEnd.y),
                    Math.abs(draggingEnd.x - draggingInit.x), Math.abs(draggingEnd.y - draggingInit.y));
      }

      g.setFont(smallFont);
      g.setColor(Color.red);
      if (bestIndividual != null)
      {
         g.drawString("Mejor individuo: " + bestIndividual.toString() + " [" + motor.getNonScaledFitness(bestIndividual) + "]", 0, 12);
         g.drawString("Iteracion N:    " + motor.getTotIteraciones(), 0, 24);
      }
      if (isDeJong)
      {
         g.drawString("Perf. Off Line:  " + ( (GeneracionSimpleDeJong) motor.getGeneracion()).getOffLinePerformance(), 0, 36);
         g.drawString("Perf. On Line :  " + ( (GeneracionSimpleDeJong) motor.getGeneracion()).getOnLinePerformance(), 0, 48);
      }

      double width = getWidth() - 50;
      double height = getHeight() - 50;

      double pixelSizeX = width / (rangeDer - rangeIzq);
      double pixelSizeY = height / (rangeTop - rangeBottom);

      g.setColor(Color.gray);
      g.setFont(bigFont);

      int lineVert;
      if (rangeIzq < 0 && rangeDer > 0)
      {
         lineVert = 0;
      }
      else
      {
         lineVert = (int) (rangeDer + rangeIzq) / 2;
      }
      g.drawLine( (int) ( (lineVert - rangeIzq) * pixelSizeX + 25), 0, (int) ( (lineVert - rangeIzq) * pixelSizeX + 25), getHeight());
      g.drawString("" + lineVert, (int) ( (lineVert - rangeIzq) * pixelSizeX + 30), (int) height + 25);
      if (rangeIzq == rangeDer)
      {
         g.drawString("Poblacion contenida en x = " + rangeIzq, 0, (int) (height / 2) - 40);
      }

      int lineHoriz = 0;
      if (rangeBottom < 0 && rangeTop > 0)
      {
         lineHoriz = 0;
      }
      else
      {
         lineHoriz = (int) (rangeTop + rangeBottom) / 2;
      }
      if (rangeBottom == rangeTop)
      {
         g.drawString("Poblacion contenida en y = " + rangeTop, 0, (int) (height / 2) - 20);
      }

      g.drawLine(0, (int) ( (rangeBottom - lineHoriz) * pixelSizeY + height + 25), getWidth(), (int) ( (rangeBottom - lineHoriz) * pixelSizeY + height + 25));
      g.drawString("" + lineHoriz, 5, (int) ( (rangeBottom - lineHoriz) * pixelSizeY + height + 20));

      g.setColor(Color.blue);
      for (Enumeration enum = currIndividuals.elements(); enum.hasMoreElements(); )
      {
         IIndividuo individuo = (IIndividuo) enum.nextElement();
         double x = getXValue(individuo);
         double y = motor.getNonScaledFitness(individuo);
         g.drawRect( (int) ( (x - rangeIzq) * pixelSizeX + 25), (int) ( (rangeBottom - y) * pixelSizeY + height + 25), (int) (pixelWidth), (int) (pixelHeight));
      }
      paintFixAxis(g);
   }

   private boolean fixAxis = false;
   private static int checkSize = 12;
   private static Color checkColor = Color.red.darker();

   private void paintFixAxis(Graphics g)
   {
      int x = 0;
      int y = getHeight() - checkSize - 3;
      g.setColor(checkColor);
      g.drawRect(x, y, checkSize, checkSize);
      if (fixAxis)
      {
         g.drawLine(x, y, x + checkSize, y + checkSize);
         g.drawLine(x, y + checkSize, x + checkSize, y);
      }
      g.drawString("Fijar grilla", x + checkSize + 2, y + 11);
   }

   static double pixelWidth = 1; // Tamao de cada punto
   static double pixelHeight = 1;

   /***************************
    * Interfaz IMotorListener *
    ***************************/
   public void inicioIteracion()
   {
      if (!running)
      {
         return;
      }
      showMsg("Inicio iteracion...");
   }

   private double getXValue(IIndividuo individuo)
   {
      if (individuo instanceof IndividuoMultipleFuncionExpresion)
      { // Si se trata de una funcion de varias variables
         return ( (IFuncion) ( (IndividuoMultipleFuncionExpresion) individuo).getParameter(curVarIndependiente)).value();
      }
      else
      {
         return ( (IFuncion) individuo).value();
      }
   }

   double rangeIzq, rangeDer, rangeTop, rangeBottom;
   Vector currIndividuals = new Vector();
   int curVarIndependiente = 0;

   public void setCurVarIndependiente(int curVarIndependiente)
   {
      if(curVarIndependiente < 0)return;
      if (this.curVarIndependiente != curVarIndependiente)
      {
         this.curVarIndependiente = curVarIndependiente;
         procesoIteracion();
      }
      else
      {
         this.curVarIndependiente = curVarIndependiente;
      }
      repaint();
   }

   public void finIteracion()
   {
      if (running)
      {
         procesoIteracion();
      }
   }

   public void procesoIteracion()
   {
      if (motor == null)
      {
         return;
      }
      bestIndividual = motor.getBestIndividuo();
      showMsg("Fin iteracion...");
      if (!fixAxis)
      {
         rangeDer = rangeTop = -Double.MAX_VALUE;
         rangeIzq = rangeBottom = Double.MAX_VALUE;
      }
      currIndividuals.removeAllElements();
      // Ahora busco el rango de accion en el eje X y en el Y
      for (Enumeration enum = motor.getPoblacion().elements(); enum.hasMoreElements(); )
      {
         IIndividuo individuo = (IIndividuo) enum.nextElement();
         if (!fixAxis)
         {
            double x = getXValue(individuo);
            double y = motor.getNonScaledFitness(individuo);
            if (x < rangeIzq)
            {
               rangeIzq = x;
            }
            if (x > rangeDer)
            {
               rangeDer = x;
            }
            if (y < rangeBottom)
            {
               rangeBottom = y;
            }
            if (y > rangeTop)
            {
               rangeTop = y;
            }
         }
         currIndividuals.addElement(individuo);
      }

      paint(super.getGraphics());
//    repaint();
   }

   public void finAlgoritmo()
   {
      if (!running)
      {
         return;
      }
//    showMsg("Fin Algoritmo...");
   }

   /** Resetea la grafica */
   public void reset()
   {
      running = false;
      repaint();
   }

   protected void processDrag()
   {
      double[] a = getPoint(draggingInit);
      double[] b = getPoint(draggingEnd);
      rangeIzq = a[0] < b[0] ? a[0] : b[0];
      rangeDer = a[0] > b[0] ? a[0] : b[0];
      rangeBottom = a[1] < b[1] ? a[1] : b[1];
      rangeTop = a[1] > b[1] ? a[1] : b[1];
      repaint();
   }

   private double[] getPoint(Point scrPoint)
   {
      double width = getWidth() - 50;
      double height = getHeight() - 50;

      double posX = ( (width - scrPoint.x + 25) * rangeIzq + (scrPoint.x - 25) * rangeDer) / width;
      double posY = ( (height - scrPoint.y + 25) * rangeTop + (scrPoint.y - 25) * rangeBottom) / height;
      return new double[]
          {posX, posY};
   }

   private int calcDistance(Point a, Point b)
   {
      return (int) Math.sqrt(Math.pow(a.x - b.x, 2) + Math.pow(a.y - b.y, 2));
   }

   /******************************
    * Interfaz ComponentListener *
    ******************************/
   public void componentHidden(ComponentEvent evt)
   {
      ;
   }

   public void componentShown(ComponentEvent evt)
   {
      ;
   }

   public void componentMoved(ComponentEvent evt)
   {
      ;
   }

   public void componentResized(ComponentEvent evt)
   {
      screenWidth = (int)super.getSize().width;
      screenHeight = (int)super.getSize().height;
   }

   /******************************
    * Interfaz MouseListener     *
    ******************************/
   public void mouseExited(MouseEvent evt)
   {
   }

   public void mouseEntered(MouseEvent evt)
   {
   }

   Point draggingInit = new Point(0, 0);
   Point draggingEnd = new Point(0, 0);
   boolean dragging = false;
   boolean isZoomIn = false;

   public void mousePressed(MouseEvent evt)
   {
      draggingInit = evt.getPoint();
      isZoomIn = (evt.getModifiers() & MouseEvent.BUTTON1_MASK) != 0;
      if (!isZoomIn)
      {
         statusBar.showMsg("Aplicando Zoom Out");
         double zoomWidth = (rangeDer - rangeIzq) / 3;
         double zoomHeight = (rangeTop - rangeBottom) / 3;
         rangeIzq -= zoomWidth;
         rangeDer += zoomWidth;
         rangeTop += zoomHeight;
         rangeBottom -= zoomHeight;
         repaint();
      }
   }

   public void mouseReleased(MouseEvent evt)
   {
      if (dragging)
      {
         isZoomIn = false;
         dragging = false;
         processDrag();
      }
   }

   public void mouseClicked(MouseEvent evt)
   {
      if (evt.getX() <= checkSize && evt.getY() > getHeight() - 2 * checkSize)
      {
         fixAxis = !fixAxis;
         repaint();
         if (fixAxis)
         {
            showMsg("Se ha dejado la Grilla fija");
         }
         else
         {
            showMsg("Se ha dejado la Grilla mvil");
         }
         return;
      }
      if ( (evt.getModifiers() & MouseEvent.BUTTON1_MASK) != 0)
      {
         pixelWidth %= 5;
         pixelWidth++;

         pixelHeight %= 5;
         pixelHeight++;
         repaint();
      }
   }

   /********************************
    * Interfaz MouseMotionListener *
    ********************************/

   public void mouseMoved(MouseEvent evt)
   {
      if (motor == null)
      {
         return;
      }
      mouseLocation = evt.getPoint();
      double width = getWidth() - 50;
      double height = getHeight() - 50;

      double posX = ( (width - mouseLocation.x + 25) * rangeIzq + (mouseLocation.x - 25) * rangeDer) / width;
      double posY = ( (height - mouseLocation.y + 25) * rangeTop + (mouseLocation.y - 25) * rangeBottom) / height;

      statusBar.showMsg("Mouse: " + posX + " , " + posY);
   }

   public void mouseDragged(MouseEvent evt)
   {
      if (!isZoomIn || calcDistance(evt.getPoint(), draggingInit) < 10)
      {
         return;
      }
      dragging = true;
      Point lastDraggingEnd = draggingEnd;
      draggingEnd = evt.getPoint();
      repaint(Math.min(Math.min(draggingInit.x, draggingEnd.x), lastDraggingEnd.x) - 5, Math.min(Math.min(draggingInit.y, draggingEnd.y), lastDraggingEnd.y) - 5,
              Math.max(Math.abs(draggingEnd.x - draggingInit.x), Math.abs(lastDraggingEnd.x - draggingInit.x)) + 10,
              Math.max(Math.abs(draggingEnd.y - draggingInit.y), Math.abs(lastDraggingEnd.y - draggingInit.y)) + 10);
      statusBar.showMsg("Aplicar Zoom In: (" + draggingInit.x + ", " + draggingInit.y + ") - (" + draggingEnd.x + ", " + draggingEnd.y + ")");
   }
}