package com.brownsoft.util;

/**
 *
 * @author Gustavo Brown
 * @since JDK 1.0
 * @version 1.1
 *
 */

import java.util.Vector;
import java.util.Hashtable;
import java.util.Enumeration;
import java.util.zip.*;
import java.io.*;

/**
 ** Esta clase realiza el parseo de un .INI
 ** Tiene manejo de secciones
 ** El formato del archivo es el siguiente
 ** 'Seccion general'  -> Es la seccion sin nombre
 **  Property1= Value1
 **  Property2= Value2
 **      .
 **      .
 **  PropertyN= ValueN
 **
 **  [Seccion 1]
 **  Property1= Value1
 **  Property2= Value2
 **      .
 **      .
 **  PropertyN= ValueN
 ** [Seccion N]
 **
 ** Se pueden agregar varios Values en una property con 'addItemToProperty' y se puede obtener la lista de Values
 ** a un solo Property con el mtodo estatico parseLine
 **
 * @author Gustavo Brown
 * @version 1.0
 */
public class ParseINI
{
   private static final int MAX_LINE_LENGTH = 255; // Maximo largo de una property
   private static final byte SECTION_SEPARATOR_CHAR = (byte) '&'; // Separador interno de secciones
   private static final String SECTION_SEPARATOR = "&";
   private static final String GENERAL = "&General&"; // Seccion General

   private String entryName;
   private DataInputStream inputStream;
   private Hashtable sections = new Hashtable();
   private Hashtable sectionEntries;
   private Hashtable general;
   private Hashtable aliased = new Hashtable(); // Contiene las dulpas (original -> alias)
   private Hashtable alias = new Hashtable(); // Contiene las dulpas (alias -> original)
   private boolean need2Save = false, autoSave = true;
   private String filename = null;

   byte[] tempBytes = new byte[MAX_LINE_LENGTH];

   public ParseINI()
   {
      general = new Hashtable();
      filename = null;
   }

   public ParseINI(String filename) throws IOException
   {
      this.filename = new File(filename).getAbsolutePath();
      try
      {
         FileInputStream inputStream = new FileInputStream(filename);
         load(inputStream);
         inputStream.close();
      }
      catch (FileNotFoundException fnfe)
      { // Si debo crear el archivo
         (new FileWriter(filename)).close(); // Creo el archivo
         general = new Hashtable();
      }
   }

   /**
    * @param inputStream DataInputStream de donde obtener el INI
    */
   public ParseINI(InputStream inputStream) throws IOException
   {
      load(inputStream);
   }

   /**
    * @param inBytes Array de bytes de donde obtener elINI
    */
   public ParseINI(byte inBytes[]) throws IOException
   {
      load(new ByteArrayInputStream(inBytes));
   }

   /**
    * @param zipFile ZipFile
    * @param entry entry que contiene la informacion del INI
    */
   public ParseINI(ZipFile zipFile, ZipEntry entry) throws IOException
   {
      if (entry == null)
      {
         return;
      }
      DataInputStream in = new DataInputStream(zipFile.getInputStream(entry));
      byte[] tempBytes = new byte[ (int) entry.getSize()];
      in.readFully(tempBytes);
      load(new ByteArrayInputStream(tempBytes));
   }

   /** Indica si se ha modificado el INI
    * @return true si se ha modificado el .INI
    */
   public boolean need2Save()
   {
      return need2Save;
   }

   /** Guardo nuevamente el .INI si este se ha modificado
    */
   public void finalize()
   {
      if (autoSave && need2Save)
      {
         try
         {
            save();
         }
         catch (IOException e)
         {
            System.out.println("ParseINI (" + filename + "):" + e);
         }
      }
   }

   /** Devuelve un Enumeration con los Hashtables conteniendo los entries de
    * cada seccion
    */
   public Enumeration sectionElements()
   {
      return sections.elements();
   }

   /** Devuelve un Enumeration con los nombres de las secciones
    */
   public Enumeration sectionNames()
   {
      return sections.keys();
   }

   /** Devuelve una copia del Hashtable conteniendo la seccion especificada
    * @param section Nombre de la seccion
    * @return Hashtable con copia de los entries de esa seccion
    */
   public Hashtable getSection(String section)
   {
      if (sections.containsKey(section))
      {
         return (Hashtable) ( (Hashtable) sections.get(section)).clone();
      }
      else
      {
         return null;
      }

   }

   /** Indica si existe una seccion
    *  @param section Nombre de la seccin
    *  @return boolean True si la seccin existe
    */
   public boolean sectionExists(String section)
   {
      return sections.containsKey(section);
   }

   /** Elimina una seccion
    * @param section Nombre de la seccion a eliminar
    */
   public void removeSection(String section)
   {
      need2Save |= sections.remove(section) != null;
   }

   /** Elimina una property de una seccion
    * @param section Nombre de la seccion
    * @param prop Nombre de la property a eliminar
    */
   public void removeProperty(String section, String prop)
   {
      if (sections.containsKey(section))
      {
         need2Save |= ( (Hashtable) sections.get(section)).remove(prop) != null;
      }
   }

   /** Elimina una property General
    * @param prop Nombre de la property a eliminar
    */
   public void removeGeneralProperty(String prop)
   {
      need2Save |= general.remove(prop) != null;
   }

   /** Devuelve la property asociada
    * @param prop Nombre de la property a obtener
    * @return el value de la property o null si no existe es property
    */
   public String getGeneralProperty(String prop)
   {
      return getProperty(general, prop, null);
   }

   /** Devuelve la property asociada en la seccion General
    * @param prop Nombre de la property a obtener
    * @param defecto String a retornar si no se encuentra esa entrada
    * @return el value de la property o el string por defecto si no existe es property
    */
   public String getGeneralProperty(String prop, String defecto)
   {
      return getProperty(general, prop, defecto);
   }

   /** Agrega una General property
    * @param prop Nombre de la property
    * @param value Valor de la property
    */
   public void setGeneralProperty(String prop, String value)
   {
      need2Save |= !value.equals(general.put(prop, value));
   }

   /** Agrega una generalProperty solo si no exista
    * @param prop Nombre de la property
    * @param value Valor de la property
    * @return true si se agrego la GeneralProperty y false si ya exista
    */
   public boolean setupGeneralProperty(String prop, String value)
   {
      if (general.containsKey(prop))
      {
         return false;
      }
      else
      {
         setGeneralProperty(prop, value);
      }
      return true;
   }

   /** Agrega una property a una seccion ( y crea la seccion en caso de que
    * no exista)
    * @param section Nombre de la seccion
    * @param prop Nombre de la property
    * @param value Valor de la property
    */
   public void setProperty(String section, String prop, String value)
   {
      if (!sections.containsKey(section))
      {
         sections.put(section, new Hashtable());
      }
      need2Save |= !value.equals( ( (Hashtable) sections.get(section)).put(prop, value));
   }

   /** Devuelve la property asociada de la seccin especificada
    * @param section Nombre de la seccin
    * @param prop Nombre de la property a obtener
    * @return el value de la property o null si no existe es property
    */
   public String getProperty(String section, String prop)
   {
      return getProperty( (Hashtable) sections.get(section), prop, null);
   }

   /** Devuelve la property asociada de la seecin especificada
    * @param section Nombre de la seccin
    * @param prop Nombre de la property a obtener
    * @param defecto String a retornar si no se encuentra esa entrada
    * @return el value de la property o el string por defecto si no existe es property
    */
   public String getProperty(String section, String prop, String defecto)
   {
      return getProperty( (Hashtable) sections.get(section), prop, defecto);
   }

   private String getProperty(Hashtable section, String prop, String defecto)
   {
      if (section != null && section.containsKey(prop))
      {
         return (String) section.get(prop);
      }
      if (section != null && alias.containsKey(prop)) // veo si es un alias
      {
         return getProperty(section, (String) alias.get(prop), defecto);
      }
      return defecto;
   }

   /** Agrega un item a una property con muli-items (separados por un 'separador')
    * @param section Nombre de la seccin
    * @param prop Nombre de la property a agregar item
    * @param item Item a agregar
    * @param separator separador
    * @return true si el item se agrego y false si ya exista ese item en la property
    */
   public boolean addItemToProperty(String section, String prop, String item, String separator)
   {
      Vector tempVector = parseLine(getProperty(section, prop), separator);
      if (tempVector.contains(item))
      {
         return false;
      }
      tempVector.addElement(item);
      String temp = "";
      for (Enumeration enum = tempVector.elements(); enum.hasMoreElements(); )
      {
         temp += (String) enum.nextElement() + separator;
      }
      setProperty(section, prop, temp);
      return true;
   }

   /** Agrega un item a una General property con muli-items (separados por un 'separador')
    * @param prop Nombre de la property a agregar item
    * @param item Item a agregar
    * @param separator separador
    * @return true si el item se agrego y false si ya exista ese item en la property
    */
   public boolean addItemToGeneralProperty(String prop, String item, String separator)
   {
      Vector tempVector = parseLine(getGeneralProperty(prop), separator);
      if (tempVector.contains(item))
      {
         return false;
      }
      tempVector.addElement(item);
      String temp = "";
      for (Enumeration enum = tempVector.elements(); enum.hasMoreElements(); )
      {
         temp += (String) enum.nextElement() + separator;
      }
      setGeneralProperty(prop, temp);
      return true;
   }

   /** Elimina un item a una property con muli-items (separados por un 'separador')
    * @param section Nombre de la seccin
    * @param prop Nombre de la property a eliminar item
    * @param item Item a eliminar
    * @param separator separador
    * @return true si el item se elimino y false si no existi ese item
    */
   public boolean removeItemFromProperty(String section, String prop, String item, String separator)
   {
      Vector tempVector = parseLine(getProperty(section, prop), separator);
      if (!tempVector.contains(item))
      {
         return false;
      }
      tempVector.removeElement(item);
      String temp = "";
      for (Enumeration enum = tempVector.elements(); enum.hasMoreElements(); )
      {
         temp += (String) enum.nextElement() + separator;
      }
      if (temp.equals(""))
      {
         removeProperty(section, prop);
      }
      else
      {
         setProperty(section, prop, temp);
      }
      return true;
   }

   /** Elimina un item a una General Property con muli-items (separados por un 'separador')
    * @param section Nombre de la seccin
    * @param prop Nombre de la property a eliminar item
    * @param item Item a eliminar
    * @param separator separador
    * @return true si el item se elimino y false si no existi ese item
    */
   public boolean removeItemFromGeneralProperty(String prop, String item, String separator)
   {
      Vector tempVector = parseLine(getGeneralProperty(prop), separator);
      if (!tempVector.contains(item))
      {
         return false;
      }
      tempVector.removeElement(item);
      String temp = "";
      for (Enumeration enum = tempVector.elements(); enum.hasMoreElements(); )
      {
         temp += (String) enum.nextElement() + separator;
      }
      if (temp.equals(""))
      {
         removeGeneralProperty(prop);
      }
      else
      {
         setGeneralProperty(prop, temp);
      }
      return true;
   }

   /** Setea un alias a una seccion
    * @param alias alias de la seccion
    * @param original nombre original de la seccion
    * @return true si la operacin se realizo con xito
    */
   public boolean setAlias(String alias, String original)
   {
      if (!sections.containsKey(original))
      {
         return false;
      }
      else
      {
         aliased.put(original, alias); // original -> alias
         this.alias.put(alias, original); // alias -> original
         return true;
      }
   }

   /** Renombra una seccion
    * @param nuevo nuevo nombre de la seccion
    * @param original nombre original de la seccion
    *
    * @return true si la operacin se realizo con xito
    */
   public boolean renameSection(String nuevo, String original)
   {
      if (!sections.containsKey(original) || sections.containsKey(nuevo))
      {
         return false;
      }
      else
      if (!original.equals(nuevo))
      {
         sections.put(nuevo, (Hashtable) sections.get(original));
         sections.remove(original);
         need2Save = true;
      }
      return true;
   }

   /** True si la seccion tiene un alias
    * @param original Nombre de la seccion
    * @return True si la seccion tiene un alias
    */
   public boolean isAliased(String original)
   {
      return aliased.containsKey(original);
   }

   /** Devuelve el alias de una seccion
    * @param original Nombre de la seccion a obtener su alias
    * @return Alias de la seccion o nombre original si no existe alias
    */
   public String getAlias(String original)
   {
      if (isAliased(original))
      {
         return (String) aliased.get(original);
      }
      else
      {
         return original;
      }
   }

   /** Carga de un DataInputStream las properties
    * Las seccin se setean con la property 'Name: SectionName' o '[SectionName]'
    * y termina cuando aparece otro inicio de seccin
    * Si no se especifica ninguna seccion se toma una por defecto
    * @param inputStream el DataInputStream
    */
   public void load(InputStream iStream) throws IOException
   {
      inputStream = new DataInputStream(iStream);

      sectionEntries = new Hashtable();
      sections.put(GENERAL, sectionEntries);
      try
      {
         while ( (entryName = readEntryName()) != null)
         {
            sectionEntries.put(entryName, readEntry());
         }
      }
      catch (EOFException e)
      {}
      general = (Hashtable) sections.get(GENERAL);
      sections.remove(GENERAL);
   }

   /** Guarda el INI en el archivo que se abrio este INI
    */
   public void save() throws IOException
   {
      if (need2Save && filename != null)
      {
         save(new FileOutputStream(filename));
      }
   }

   /** Setea que si se quiere actualizar el archivo del INI automaticamente
    * @param autoSave True si se quiere actualizar automaticamente el archvo del INI
    */
   public void setAutoSave(boolean autoSave)
   {
      this.autoSave = autoSave;
   }

   /** Enva una serializacin de una seccion al outputStream pasado como parmetro
    * @param section Nombre de la seccin a serializar
    * @param out OutputStream por donde mandar la seccion
    * @param sendGeneral True si se desea envar las General Properties
    * @return True si la seccion existia
    */
   public boolean serializeSection(String section, OutputStream out, boolean sendGeneral) throws IOException
   {

      if (!sections.containsKey(section))
      {
         return false;
      }
      ObjectOutputStream serialOut = new ObjectOutputStream(out);
      serialOut.writeObject(section);
      out.flush();
      serialOut.writeObject(sections.get(section));
      out.flush();
      serialOut.writeObject(new Boolean(sendGeneral));
      out.flush();
      if (sendGeneral)
      {
         serialOut.writeObject(general);
      }
      out.flush();
      return true;
   }

   /** Recibe una serializacin de una seccion y la agraga al INI
    * @param in InputStream de donde obtener la seccin
    * @return section Nombre de la seccion recibida
    */
   public String unserializeSection(InputStream in) throws IOException, ClassNotFoundException
   {
      ObjectInputStream serialIn = new ObjectInputStream(in);
      String section = (String) serialIn.readObject();
      Hashtable loadedSection = (Hashtable) serialIn.readObject();
      if (!sections.containsKey(section) || ! ( (Hashtable) sections.get(section)).equals(loadedSection))
      { // Si la seccion no existia o si la seccion leda es diferente a la que ya existia
         sections.put(section, loadedSection);
         need2Save = true;
      }
      if ( ( (Boolean) serialIn.readObject()).booleanValue())
      {
         general = (Hashtable) serialIn.readObject();
      }

      return section;
   }

   /** Guarda el INI
    * @param oStrem el Outputstream
    */
   public void save(OutputStream oStream) throws IOException
   {
      DataOutputStream outputStream = new DataOutputStream(oStream);
      Enumeration props, secs;
      String prop, value;
      int it;
      if (general != null)
      {
         props = general.keys();
         while (props.hasMoreElements())
         {
            prop = (String) props.nextElement();
            if ( (value = (String) getGeneralProperty(prop)).equals(""))
            {
               continue;
            }
            if ( (value.length() + prop.length() + 4) > MAX_LINE_LENGTH)
            {
               it = (MAX_LINE_LENGTH - prop.length() - 4);
            }
            else
            {
               it = value.length();
            }
            outputStream.writeBytes(prop + "= " + value.substring(0, it) + "\r\n");
            for (; (value.length() - it) > (MAX_LINE_LENGTH - 4); it += MAX_LINE_LENGTH)
            {
               outputStream.writeBytes(" " + value.substring(it, it + MAX_LINE_LENGTH) + "\r\n");
            }
            if (it < value.length())
            {
               outputStream.writeBytes(" " + value.substring(it) + "\r\n");
            }
         }
      }
      secs = sectionNames();
      while (secs.hasMoreElements())
      {
         String section = (String) secs.nextElement();
         outputStream.writeBytes("\r\n[" + section + "]\r\n");
         props = ( (Hashtable) sections.get(section)).keys();
         while (props.hasMoreElements())
         {
            prop = (String) props.nextElement();
            if ( (value = (String) getProperty(section, prop)).equals(""))
            {
               continue;
            }
            if ( (value.length() + prop.length() + 4) > MAX_LINE_LENGTH)
            {
               it = (MAX_LINE_LENGTH - prop.length() - 4);
            }
            else
            {
               it = value.length();
            }
            outputStream.writeBytes(prop + "= " + value.substring(0, it) + "\r\n");
            for (; (value.length() - it) > (MAX_LINE_LENGTH - 4); it += MAX_LINE_LENGTH)
            {
               outputStream.writeBytes(" " + value.substring(it, it + MAX_LINE_LENGTH) + "\r\n");
            }
            if (it < value.length())
            {
               outputStream.writeBytes(" " + value.substring(it) + "\r\n");
            }
         }
      }
      outputStream.close();
      need2Save = false;
   }

   /** Obtiene del InputStream el nombre del entry
    * @param inputStream el DataInputStream
    * @return el nombre del entry
    */
   private String readEntryName() throws IOException
   {
      int offset = 0;
      byte car;

      while (true)
      {
         switch ( (char) (car = inputStream.readByte()))
         {
            case '%': // Comentarios
            case '*':
               while (true)
               {
                  car = inputStream.readByte();
                  if (car == '\n' || car == '\r')
                  {
                     break;
                  }
               }
               break;
            case ' ': // Continuacin del entry anterior -> se concatena con el entry que ya se estaba procesando
               if (entryName == null)
               {
                  throw new IOException("Invalid entry");
               }
               sectionEntries.put(entryName, (String) sectionEntries.get(entryName) + readEntry());
               return readEntryName();
            case '[': // Comienza nueva seccion
               sectionEntries = new Hashtable();
               String sectionName = readEntry();
               sectionName = sectionName.substring(0, sectionName.length() - 1);
               sections.put(sectionName, sectionEntries);
               entryName = null;
               return readEntryName();
            case '\n':
            case '\r':
               break;
            default: // Tomo entry
               tempBytes[offset++] = car;
               while (true)
               {
                  car = inputStream.readByte();
                  if ( /*Character.isWhitespace((char)car) ||*/
                      car == ':' || car == '=')
                  {
                     break;
                  }
                  else
                  {
                     tempBytes[offset++] = car;
                  }
               }
               if (offset == 4 && // Chequeo si es nueva Seccion
                   tempBytes[0] == 'N' &&
                   tempBytes[1] == 'a' &&
                   tempBytes[2] == 'm' &&
                   tempBytes[3] == 'e')
               {
                  sectionEntries = new Hashtable();
                  String section = readEntry();
                  if (section.startsWith("./"))
                  {
                     section = section.substring(2);
                  }
                  sections.put(section, sectionEntries);
                  entryName = null;
                  return readEntryName();
               }
               else
               {
                  return new String(tempBytes, 0, offset);
               }
         }
      }
   }

   /** Obtiene del InputStream el value del entry
    * @param inputStream el DataInputStream
    * @return el value del entry
    */
   private String readEntry() throws IOException
   {
      byte[] b = new byte[MAX_LINE_LENGTH];
      int offset = 0;
      byte car;
      while (Character.isWhitespace( (char) (car = inputStream.readByte())))
      {
         ;
      }
      b[offset++] = car;
      try
      {
         while ( (car = inputStream.readByte()) != '\0' && car != '\n' && car != '\r')
         {
            b[offset++] = car;
         }
      }
      catch (EOFException e)
      {}
      return new String(b, 0, offset);
   }

   // Mtodos de clase

   /** Remueve del string los espacios al comienzo y deja solo un espacio
    * cuando hay varios seguidos
    */
   public static String removeExtraSpaces(String line)
   {

      int index, offset, lineLength;
      if (line == null)
      {
         return null;
      }
      lineLength = line.length();
      char b[] = new char[lineLength];
      if (lineLength == 0)
      {
         return line;
      }
      for (index = 0; index < lineLength; index++)
      {
         if (line.charAt(index) != ' ')
         {
            break;
         }
      }
      for (offset = 0; index < lineLength - 1; index++)
      {
         if (line.charAt(index) != ' ' || line.charAt(index + 1) != ' ')
         {
            b[offset++] = line.charAt(index);
         }
      }
      b[offset++] = line.charAt(index);
      return new String(b, 0, offset);
   }

   /** Separa partes de una lines y las retorna en un Vector
    * @param line linea a separar
    * @param separator String conteniendo el separador de partes
    * @return Vector de Strings conteniendo las partes separadas
    */
   public static Vector parseLine(String line, String separator)
   {
      Vector partes = new Vector();
      int index = 0, offset = 0;
      int indexComillas;
      boolean startingComillas = true;
      if (line == null)
      {
         return partes;
      }
      if (!line.endsWith(separator))
      {
         line += separator;
      }
      if ( (indexComillas = line.indexOf('\"')) == -1)
      {
         indexComillas = Integer.MAX_VALUE;
      }
      while ( (index = line.indexOf(separator, startingComillas ? offset : indexComillas)) != -1)
      {
         if (index > indexComillas)
         {
            if ( (indexComillas = line.indexOf('\"', index)) == -1)
            {
               indexComillas = Integer.MAX_VALUE;
            }
            if (startingComillas)
            {
               startingComillas = false;
               offset++;
               if (indexComillas == Integer.MAX_VALUE)
               {
                  break;
               }
               else
               {
                  continue;
               }
            }
            else
            {
               startingComillas = true;
            }
            index--;
         }
         partes.addElement(line.substring(offset, index));
         offset = index;
         while (line.startsWith(separator, ++offset) && offset < line.length())
         {
            ; // Elimino separadores seguidos
         }
      }
      if (!startingComillas) // Si faltan las comillas de cierre, igual pongo esa parte
      {
         partes.addElement(line.substring(line.length() - separator.length()));
      }
      return partes;
   }
}
