package com.brownsoft.codec;
import java.io.*;
import java.awt.*;
import java.awt.image.*;

/**
 * <p>Title: Proyecto Codificacion de Imagenes y Video</p>
 * <p>Description: </p>
 * Esta clase implementa un decodificador de imgenes a color y B/N aplicando la
 * transformacin de Karhunen-Love
 * <p>Copyright: Copyright (c) 2004</p>
 * <p>Company: </p>
 * @author Gustavo Brown (alegus@adinet.com.uy)
 * @version 1.0
 */

public class KLTDecoder implements KLTConstants
{
   private boolean ownerInputStream = false;
   private DataInputStream dataStream;

   private String comment = "";
   private int quality;
   private int coefficients;
   private boolean useQuantization;
   private boolean embedKLTVectors;
   private byte imageType;
   private int imageWidth, imageHeight;
   private BufferedImage image;
   private HuffmanMultiplexerBitInputStream huffmanDecoder;
   private HuffmanMultiplexerBitInputStream externalKLTReader;
   private double[][][] imageMatrix;
   private int currentBand;
   protected int BLOCK_WIDTH;
   protected int BLOCK_SIZE;
   private boolean showBands = false;
   private boolean useExternalKLTBase = false;
   private String externalKLTBaseInputName;


//**********************************
// Intefaz pblica
//**********************************

   /** Construye una instance del decoder
    * @param filename Nombre del archivo donde se encuentra la imagen
    */
   public KLTDecoder(String filename)throws IOException
   {
      this(new FileInputStream(filename));
      ownerInputStream = true;
   }

   /** Construye una instance del decoder
    * @param inputStream Stream de donde obtener la imagen
    */
   public KLTDecoder(InputStream inputStream)throws IOException
   {
	   dataStream = new DataInputStream(inputStream);
       readHeader();
   }

	/** Indica si se desea mostrar las bandas que se van codificando */
    public void setShowBands(boolean showBands)
    {
       this.showBands = showBands;
    }

	/** Retorna si se mostraran las bandas al codificar la imagen */
    public boolean getShowBands()
    {
       return showBands;
    }

    /** Obtiene el comentario asociado a la imagen
     * @return Comentario asociado a la imagen
     */
    public String getComment()
    {
       return comment;
    }


    /** Obtiene el tamao del bloque NxN
     * return Tamao del bloque NxN
     */
    public int getBlockSize()
    {
       return BLOCK_WIDTH;
    }

    /** Obtiene el tipo de imagen
     * @return tipo de imagen (IMAGE_GRAY, IMAGE_RGB, IMAGE_YCrCb)
     */
    public int getImageType()
    {
       return imageType;
    }

    /** Obtiene el tamao de la imagen
     * @return tamao de la imagen
     */
    public Point getImageSize()
    {
       return new Point(imageWidth, imageHeight);
    }

    /** Indica si los coeficientes de la base de la KLT estan embebidos en la imagen
     * @param booleano indicando si los coeficientes de la base estn embebidos
     */
    public boolean getKLTVectorsEmbeded()
    {
       return embedKLTVectors;
    }

   /** Indica que utilice la base de la klt a partir del archivo pasado por argumento
    * @param externalKLTBase Nombre del archivo donde obtener la base de la KLT
    */
   public void setKLTBaseName(String kltBaseName)
   {
      useExternalKLTBase = true;
      externalKLTBaseInputName = kltBaseName;
   }

   /** Decodifica la imagen
    * @return Image con la imagen
    */
   public Image decode() throws KLTException
   {
      try
      {
         if (image == null)
         {
            decodeImage();
         }
         return image;
      } catch (IOException e)
      {
         e.printStackTrace();
         throw new KLTException("Ocurrio un error de IO al decodificar la imagen: " + e.getMessage());
      }
   }

//**********************************
// Mtodos privados
//**********************************

	/** Decodifica la imagen */
	private void decodeImage()throws IOException,KLTException
    {
        decompressImage();

		if(ownerInputStream)
        { // Si el stream original lo abri yo, entonces lo cierro
           dataStream.close();
        }
    }

	/** Lee el header */
    private void readHeader() throws IOException, KLTException
    {
      // Lee el Header de la imagen
      // El header esta compuesto:
      // FORMAT_STRING -> String (34 bytes) que identifica el formato de la imagen
      // COMMENT_SIZE -> Short (2 bytes) que indican la cantidad de bytes del comentario de la imagen
      // FLAGS -> 1 byte con las flags de la imagen
      //        \-> Bit 0: Indica si los vectores de la KLT estan embebidos en la imagen
      //        \-> Bit 1-2: Indica el formato de la imagen
      //          \-> 00: Imagen en escala de grises
      //              10: Imagen RGB
      //              11: Imagen YCrCb
      // BLOCK_SIZE -> 1 byte indicando el tamao(lado) de cada bloque (bloque N*N)
      // QUALITY -> 1 byte indicando la calidad de la imagen
      // IMAGE_WIDTH -> 2 bytes indicando el ancho de la imagen
      // IMAGE_HEIGHT -> 2 bytes indicando el alto de la imagen

      // Primero leo el FormatString
      byte [] tmp = new byte[FORMAT_STRING.length()];
      dataStream.readFully(tmp);
      if(!new String(tmp).equals(FORMAT_STRING))
      {
         throw new KLTException("El archivo no tiene el format correcto [FormatString]");
      }

      // Ahora leo el comentario asociado a la imagen
      short commentSize = dataStream.readShort();
      if(commentSize > 0)
      {
	      tmp = new byte[commentSize];
          dataStream.readFully(tmp);
          comment = new String(tmp);
      }

      // Ahora leo las flags
      byte flags = dataStream.readByte();
      embedKLTVectors = ((flags & 1) != 0);
      imageType = (byte)((flags >> 1) & 3);

      setBlockSize(dataStream.readByte());
      setQuality(((int)dataStream.readByte())&255);
      // Ahora leo el tamao de la imagen
      imageWidth = dataStream.readShort();
      imageHeight = dataStream.readShort();
    }

	/** Descomprime la imagen */
    private void decompressImage()throws IOException
    {
 	   // Lo primero que hago es crear la matrix que va a contener la informacion
       switch(imageType)
       {
          case IMAGE_GRAY:
             imageMatrix = new double[1][imageHeight][imageWidth];
             break;
          case IMAGE_RGB:
          case IMAGE_YCrCb:
             imageMatrix = new double[3][imageHeight][imageWidth];
       }

       huffmanDecoder = new HuffmanMultiplexerBitInputStream(dataStream);
       HuffmanMultiplexerBitInputStream kltReader = huffmanDecoder;
       if (useExternalKLTBase)
       { // Si se esta utilizando una base de la KLT externa
          openExternalKLTBase();
          kltReader = externalKLTReader;
       }
       else
       {
          if (!embedKLTVectors)
          { // Si no estan embebida la base de la KLT y no se especific un archivo externo no puedo descomprimir la imagen
             throw new KLTException("La imagen no tiene la base de la KLT embebida y no se ha especificado un archivo de coeficientes (.kltc)");
          }
       }

       // Ahora debo ir decodificando cada banda de la imagen

      int cantBands = imageMatrix.length;
      double [][][] imageBand = new double[cantBands][imageHeight][imageWidth];
      for(currentBand = 0; currentBand < cantBands; currentBand++)
      {
	     double [][] encodedImageMatrixCoeffs = new double[getCantBlocks(currentBand)][BLOCK_SIZE];
         // Ahora decodifico el bloque
         double[][] invBasis;
         double lastDC = 0;
         for(int i = 0; i < encodedImageMatrixCoeffs.length; i++)
         {
            lastDC = decodeBlock(encodedImageMatrixCoeffs[i], lastDC);
         }

         // Ahora debo obtener los vectores de la base. El source de los vectores puede
         // ser tanto el propio stream de la imagen o un stream externo
         // Lo primero que hago obtener la cantidad de coeficientes que uso
         int currentBandLeastUsedCoeff = kltReader.readUnencoded(16);
         // Y ahora voy decodificando uno a uno los versores utilizados
         invBasis = new double[BLOCK_SIZE][BLOCK_SIZE];
         for (int i = BLOCK_SIZE - 1; i >= currentBandLeastUsedCoeff; i--)
         {
            for (int j = 0; j < BLOCK_SIZE; j++)
            {
               int cantBitsNeeded = kltReader.read(DC_CHANNEL);
               invBasis[i][j] = getDoubleFromVLI(kltReader, cantBitsNeeded) / KLT_COEFF_MULTIPLIER;
            }
         }

		 if(imageType != IMAGE_YCrCb || currentBand == AXIS_Y)
         {
	         imageBand[currentBand] = KLTHelper.apply(invBasis, imageWidth, imageHeight, BLOCK_WIDTH, encodedImageMatrixCoeffs);
         }
         else
         {
            // Si la imagen esta en el espacio YCrCb debo considerar especialmente los canales Cr o Cb
            imageBand[currentBand] = KLTHelper.apply(invBasis, imageWidth / subsampling_x, imageHeight / subsampling_y, BLOCK_WIDTH, encodedImageMatrixCoeffs);
         }

         if (showBands)
         {
            // Ahora instancio un PGMDecoder con estos datos
            com.brownsoft.image.PGMEncoder decodedImage = new com.brownsoft.image.PGMEncoder(imageBand[currentBand]);
            decodedImage.showImage(getBandName());
         }
         System.gc();
      }

      // Si la imagen estaba codificada en YCrCb debo pasarla a RGB
      if(imageType == IMAGE_YCrCb)
      {
      	 imageBand = expandSubSampling(imageBand);
         imageBand = KLTHelper.convertToRGB(imageBand);
      }

       // Ahora creo el BufferedImage asociado a la imagen
       image = new BufferedImage(imageWidth, imageHeight, getBufferedImageType());
       int[] imageData = new int[imageHeight * imageWidth * cantBands];
       int k = 0;
       for (int j = 0; j < imageHeight; j++)
       {
          for (int i = 0; i < imageWidth; i++)
          {
             for (int currentBand = 0; currentBand < cantBands; currentBand++)
             {
                double value = imageBand[currentBand][j][i];
                if(value > 255)
                {
                   value = 255;
                }else if(value < 0)
                {
                   value = -value;
                }
                imageData[k++] = (int)value;
             }
          }
       }
       imageBand = null; // Ayudamos al GC
       image.getRaster().setPixels(0,0, imageWidth, imageHeight, imageData);
    }

   /** Decodifica un bloque de la imagen
    * @param vector Vector donde colocar los coeficientes del bloque
    * @param lastDC el coeficiente de continua del bloque anterior
    * @return double El coeficiente de continua actual
    */
   private double decodeBlock(double[] vector, double lastDC)throws IOException
   {
      // Primero decodifico el coeficiente de DC
      int cantBitsNeeded = huffmanDecoder.read(DC_CHANNEL); // Obtengo el size del VLI del coef de continua
      double currentDC = dequantizeDC(getDoubleFromVLI(cantBitsNeeded)) + lastDC;
      int curCoef = vector.length - 1;
      vector[curCoef--] = currentDC;

      // Ahora decodifico los coeficientes de alterna
      int cmd = huffmanDecoder.read(AC_CHANNEL_BASE + currentBand);
      while(cmd != EOB)
      {
         int zeroRunLength = (cmd >> 4);
         curCoef -= zeroRunLength;
         cantBitsNeeded = cmd & 15;
         vector[curCoef--] = dequantizeAC(getDoubleFromVLI(cantBitsNeeded), curCoef);
         cmd = huffmanDecoder.read(AC_CHANNEL_BASE + currentBand);
      }
      return currentDC;
   }

   /** Obtiene un double a partir de un VariableLengthInteger
    * @param huffmanDecoder el HuffmanMultiplexerBitInputStream con el stream de entrada
    * @param cantBitsNeeded cantidad de bits de este VLI
    * */
   private double getDoubleFromVLI(HuffmanMultiplexerBitInputStream huffmanDecoder, int cantBitsNeeded)throws IOException
   {
      int value = huffmanDecoder.readUnencoded(cantBitsNeeded);
      int cte = (1<<(cantBitsNeeded-1));
      boolean isNegative = ((value & cte) != 0);
      value &= ~cte;
      int sign = 0;

      value += cte;
      if(isNegative)
      {
         value = -value;
      }
      return (((double)value) / FRACTION_MULTIPLIER);
   }

   /** Obtiene un double a partir de un VariableLengthInteger
    * @param cantBitsNeeded cantidad de bits de este VLI
    **/
   private double getDoubleFromVLI(int cantBitsNeeded)throws IOException
   {
      return getDoubleFromVLI(huffmanDecoder, cantBitsNeeded);
   }

   /** Decuantifica el componente de continua */
   private double dequantizeDC(double value)
   {
      return value * DC_QUANTIZER;
   }

   /** Decuantifica un componente de alterna */
   private double dequantizeAC(double value, int curCoef)
   {
      if(imageType == IMAGE_YCrCb && currentBand != AXIS_Y)
      { // Si estoy decodificando en YCrCb las cromas las codifico ms grueso
	      return (value *  AC_QUANTIZER_CrCb) / (Math.pow(AC_QUANTIZER_FACTOR, curCoef) * quality) ;
      }
      else
      {
	      return (value *  AC_QUANTIZER) / (Math.pow(AC_QUANTIZER_FACTOR, curCoef) * quality) ;
      }
   }

    private int getBufferedImageType()
    {
       switch(imageType)
       {
          case IMAGE_GRAY: return BufferedImage.TYPE_BYTE_GRAY;
          case IMAGE_RGB:
          case IMAGE_YCrCb: return BufferedImage.TYPE_3BYTE_BGR;
          default: throw new KLTException("Tipo de imagen no definido: " + imageType);
       }
    }

    /** Setea el tamao(lado) de los bloques */
    public void setBlockSize(int blockSize) throws KLTException
    {
       if(blockSize < 1 || blockSize > 255)
       {
          throw new KLTException("El tamao del bloque debe estar entre 1 y 255");
       }
       this.BLOCK_WIDTH = blockSize;
       this.BLOCK_SIZE = blockSize * blockSize;
    }

	/** Retorna la cantidad de bloques (por banda) que tiene la imagen */
    private int getCantBlocks(int curBand)
    {
       int imageWidth = this.imageWidth;
       int imageHeight = this.imageHeight;
       if(imageType == IMAGE_YCrCb && curBand != AXIS_Y)
       {
          imageWidth /= subsampling_x;
          imageHeight /= subsampling_y;
       }
       int blocksWidth = (int)Math.ceil((double)imageWidth / BLOCK_WIDTH);
       int blocksHeight = (int)Math.ceil((double)imageHeight / BLOCK_WIDTH);
       return blocksWidth * blocksHeight;
    }

	/** Setea la calidad de la imagen */
    private void setQuality(int quality)
    {
    	this.quality = quality;
    }

    /** Expande el subsampling de los canales Cr y Cb
     */
    private double [][][] expandSubSampling(double [][][] imageBand)
    {
       if(imageType != IMAGE_YCrCb)
       {
          return imageBand;
       }
       double [][][] expandedImageBand = new double[3][imageHeight][imageWidth];
       expandedImageBand[AXIS_Y] = imageBand[AXIS_Y];
       for(int j = 0; j < imageHeight; j++)
       {
          for(int i = 0; i < imageWidth; i++)
          {
             expandedImageBand[AXIS_Cr][j][i] = imageBand[AXIS_Cr][j / subsampling_y][i / subsampling_x];
             expandedImageBand[AXIS_Cb][j][i] = imageBand[AXIS_Cb][j / subsampling_y][i / subsampling_x];
          }
       }
       return expandedImageBand;
    }

   /** Obtiene el nombre asociado a un canal
    * @param band Canal
    * @return Nombre asociado a un canal
    */
   private String getBandName()
   {
      switch(imageType)
      {
         case IMAGE_GRAY: return "Canal Gris";
         case IMAGE_RGB: return "Canal " + (currentBand == 0 ? "R" : currentBand == 1 ? "G": "B");
         case IMAGE_YCrCb: return "Canal " + (currentBand == 0 ? "Y" : currentBand == 1 ? "Cr": "Cb");
		 default:  return "Desconocido";
      }
   }

	/** Abre un archivo de base de la KLT (.kltc) */
    private void openExternalKLTBase() throws IOException
    {
       BufferedInputStream kltInputStream = new BufferedInputStream(new FileInputStream(externalKLTBaseInputName));
       // Primero leo el String que identifica los streams de klt
       byte[] tempBuffer = new byte[4];
       kltInputStream.read(tempBuffer);
       if (!new String(tempBuffer).equals("kltc"))
       {
          throw new KLTException("Archivo de coeficientes de la base de la KLT invlido (" + externalKLTBaseInputName + ")");
       }
       // Ahora veo si el BLOCK_WIDTH es el mismo
       int embededBlockWidth = kltInputStream.read() & 255;
       if (embededBlockWidth != BLOCK_WIDTH)
       {
          System.err.println("Warning: El archivo de coeficientes difiere en el tamao del bloque");
          System.err.println("         Se utilizara como valor " + embededBlockWidth + " en vez de " + BLOCK_WIDTH);
          setBlockSize(embededBlockWidth);
       }
       externalKLTReader = new HuffmanMultiplexerBitInputStream(kltInputStream);
    }

}