package com.brownsoft.codec;

import Jama.*;
import java.awt.*;
import java.awt.image.*;
import java.io.*;

/**
 * <p>Title: Proyecto Codificacion de Imagenes y Video</p>
 * <p>Description: </p>
 * Esta clase es un helper que trabaja con matrices. Implemento aqui varias operaciones
 * como ser aplicar la transformada de Karhunen-Love, convertir imagenes de un espacio
 * de color a otro, etc
 * <p>Copyright: Copyright (c) 2004</p>
 * <p>Company: </p>
 * @author Gustavo Brown (alegus@adinet.com.uy)
 * @version 1.0
 */

public class KLTHelper
{
   private int imageWidth, imageHeight;
   public Matrix eigenVectors;
   public Matrix invEigenVectors;
   public Matrix eigenValues;
   public Matrix pcaMatrix;
   private int BLOCK_WIDTH;
   private int BLOCK_SIZE;
   private double [][] imageMatrix;

   /** Crea una instancia de KLTHelper */
   public KLTHelper(int BLOCK_WIDTH, int BLOCK_SIZE, double [][] imageMatrix)
   {
      this.BLOCK_WIDTH = BLOCK_WIDTH;
      this.BLOCK_SIZE = BLOCK_SIZE;
      this.imageMatrix = imageMatrix;
      initialize();
   }

   public KLTHelper(int BLOCK_WIDTH, int BLOCK_SIZE, double [][] imageMatrix, double [][]invBasis)
   {
      this.BLOCK_WIDTH = BLOCK_WIDTH;
      this.BLOCK_SIZE = BLOCK_WIDTH * BLOCK_WIDTH;
      this.imageMatrix = imageMatrix;

      double[][] imageMatrixCoeffs = blockRearrange(imageMatrix);
      pcaMatrix = new Matrix(imageMatrixCoeffs);

      invEigenVectors = new Matrix(invBasis);
      eigenVectors = invEigenVectors.inverse();
   }

	/** Obtiene el array de vectores con la imagen, donde cada vector es un bloque de M*M*/
    public double [][] blockRearrange(double[][] imageMatrix)
    {
       // Primero aloco memoria para esto
       int pixelsWidth = imageMatrix[0].length;
       int pixelsHeight = imageMatrix.length;
       int blocksWidth = (int)Math.ceil((double)pixelsWidth/BLOCK_WIDTH);
       int blocksHeight = (int)Math.ceil((double)pixelsHeight/BLOCK_WIDTH);
       double [][] matrixNbyM = new double[blocksHeight*blocksWidth][BLOCK_SIZE];

		int k = 0;
		// Ahora genero la matriz
        for(int j = 0; j < blocksHeight; j++)
        {
           for(int i = 0; i < blocksWidth; i++)
           {
              matrixNbyM[k++] = getBlockAsVector(imageMatrix, i, j);
           }
        }

        return matrixNbyM;
    }

	/** Obtiene un bloque de la matriz y lo retorna como un vector*/
    public double [] getBlockAsVector(double [][] imageMatrix, int x, int y)
    {
       double [] vector = new double[BLOCK_SIZE];
       int k = 0;
	   x *= BLOCK_WIDTH;
       y *= BLOCK_WIDTH;
       int _x = x;
       for(int j = 0; j < BLOCK_WIDTH; j++)
       {
          for (int i = 0; i < BLOCK_WIDTH; i++)
          {
				vector[k++] = imageMatrix[y][x];
                if(x < imageMatrix[0].length-1)
                { // Me cambio de columna solo si no pase los limites de la imagen
                	x++;
                }
          }
          if (y < imageMatrix.length-1)
          {// Me cambio de fila si NO pase los limites de la imagen
             y++;
          }
          x = _x;
       }
       return vector;
    }

	/** A partir de un array de vectores con la imagen, obtiene la matriz de la imagen */
    public static double [][] unblockRearrange(double[][] imageMatrix, int pixelsWidth, int pixelsHeight, int BLOCK_WIDTH)
    {
       // Primero aloco memoria para esto
       double [][] matrix = new double[pixelsHeight][pixelsWidth];

       int blocksWidth = (int)Math.ceil((double)pixelsWidth/BLOCK_WIDTH);
       int blocksHeight = (int)Math.ceil((double)pixelsHeight/BLOCK_WIDTH);
       int k = 0;
       int y = 0;
       // Ahora genero la matriz
       for (int j = 0; j < blocksHeight; j++)
       {
          int _y = y;
          int x = 0;
          for (int i = 0; i < blocksWidth; i++)
          {
             double [] block = imageMatrix[k++];
             y = _y;
             int _x = x;
             for(int blockJ = 0; blockJ < BLOCK_WIDTH; blockJ++)
             {
                x = _x;
                for (int blockI = 0; blockI < BLOCK_WIDTH; blockI++)
                {
                   matrix[y][x++] = block[blockJ * BLOCK_WIDTH + blockI];
                   if(x == pixelsWidth)
                   { // Si llegue al borde de la imagen
                      break;
                   }
                }
                y++;
                if(y == pixelsHeight)
                { // Si llegue al borde de la imagen
                	break;
                }
             }
          }
       }
        return matrix;
    }

	/** Elimina los N vectores de menor peso */
	public Matrix pruneEigenVectors(Matrix vectors, int cantToLeave)
    {
       Matrix matrix = new Matrix(eigenVectors.getArrayCopy());
       int size = eigenVectors.getColumnDimension();
       int cantToPrune = size - cantToLeave;
       matrix.setMatrix(0, cantToPrune-1, 0, size-1, new Matrix(cantToPrune, size));
       return matrix;
    }

	/** Elimina los N vectores de menor peso */
	public void pruneCoeffs(double [][] matrix, int cantToLeave)
    {
       int size = matrix.length;
       int cantToPrune = BLOCK_SIZE - cantToLeave;
       double [] zeros = new double[cantToPrune];
       for(int i = 0; i < size; i++)
       {
		  System.arraycopy(zeros, 0, matrix[i], 0, zeros.length);
       }
    }

    public void initialize()
    {
        // Ahora obtengo la matriz x*y
        imageWidth = imageMatrix[0].length;
        imageHeight = imageMatrix.length;
        // La tengo que convertir en una matriz de N*8 con los elementos de la cuadricula 8x8
        double [][] imageMatrixCoeffs = blockRearrange(imageMatrix);

        // Aplico el PCA (Principal Component Analysis)
		pcaMatrix = new Matrix(imageMatrixCoeffs);
		Matrix covariance = cov(pcaMatrix);
		EigenvalueDecomposition evd = covariance.eig();
		eigenVectors = evd.getV();
		eigenValues = evd.getD();
        invEigenVectors = eigenVectors.inverse();
    }

    /** Aplica la transformacin de Karhunen-Love utilizando cantCoeffs vectores de la base
     * @param cantCoeffs cantidad de coeficientes de la base a utilizar
     */
    public double [][] applyKLTByNumberOfCoefficients(int cantCoeffs)
    {
        // Ahora aplicamos la KLT
        Matrix kltImage = pcaMatrix.times(eigenVectors);

        double [][] kltImageMatrix = kltImage.getArrayCopy();
        pruneCoeffs(kltImageMatrix, cantCoeffs);

        return kltImageMatrix;
	}

	/** Obtiene la matriz inversa de base de la KLT */
    public double[][] getInvBasis()
    {
       return invEigenVectors.getArrayCopy();
    }

    public static double [][] apply(double [][] invBasis, int imageWidth, int imageHeight, int BLOCK_WIDTH, double [][] coeffs)
    {
        Matrix invEigenVectors = new Matrix(invBasis);
        Matrix kltImageMatrix = new Matrix(coeffs);
        Matrix kltDecodedImage = kltImageMatrix.times(invEigenVectors);
        return unblockRearrange(kltDecodedImage.getArrayCopy(), imageWidth, imageHeight, BLOCK_WIDTH);
    }

	/** Obtiene la matrix de covarianza */
    public Matrix cov(Matrix inputMatrix)
    {
        int m = inputMatrix.getRowDimension();
        int n = inputMatrix.getColumnDimension();
       	Matrix X = new Matrix(n,n);
		int degrees = (m - 1);
		double c;
		double s1;
		double s2;
		for (int i = 0; i < n; i++) {
			for (int j = 0; j < n; j++) {
				c = 0;
				s1 = 0;
				s2 = 0;
				for (int k = 0; k < m; k++) {
					s1 += inputMatrix.get(k, i);
					s2 += inputMatrix.get(k, j);
				}
				s1 = s1 / m;
				s2 = s2 / m;
				for (int k = 0; k < m; k++) {
					c += (inputMatrix.get(k, i) - s1) * (inputMatrix.get(k, j) - s2);
				}
				X.set(i, j, c / degrees);
			}
		}
		return X;
    }

 	private static final double [][] RGBToYCrCbMatrix = new double [][]
						     				{{ 0.289  , 0.587  , 0.114  },
                                             { -0.1687, -0.3313, 0.5    },
                                             { 0.5    , -0.4187, -0.0813}};

 	private static final double [][] YCrCbToRGBMatrix = new double [][]
						     				{{ 1.009  , 0      , 1.417 },
                                             { 1.011  , -0.344 , -0.701},
                                             { 1.010  , 1.772  , 0.0015}};

    /** Convierte una imagen de un espacio de color a otro (RGB <--> YCrCb)
     * @param cvtMatrix matriz de conversin
     * @param imageMatrix double[][][] con la imagen a convertir
     * @return double[][][] con la matriz convertida
     */
    private static double[][][] convertColorSpace(Matrix cvtMatrix, double[][][] imageMatrix)
    {
       int imageWidth = imageMatrix[0][0].length;
       int imageHeight = imageMatrix[0].length;
       double [][] inputMatrix = new double[3][imageWidth * imageHeight];
       int k = 0;
       for(int j = 0; j < imageHeight; j++)
       {
          for(int i = 0; i < imageWidth; i++)
          {
             inputMatrix[KLTConstants.RED][k] = imageMatrix[KLTConstants.RED][j][i];
             inputMatrix[KLTConstants.GREEN][k] = imageMatrix[KLTConstants.GREEN][j][i];
             inputMatrix[KLTConstants.BLUE][k++] = imageMatrix[KLTConstants.BLUE][j][i];
          }
       }

       double [][] result = cvtMatrix.times(new Matrix(inputMatrix)).getArray();

	   double [][][]resultMatrix = new double[3][imageHeight][imageWidth];
       k = 0;
       for(int j = 0; j < imageHeight; j++)
       {
          for(int i = 0; i < imageWidth; i++)
          {
             resultMatrix[KLTConstants.RED][j][i] = result[KLTConstants.RED][k];
             resultMatrix[KLTConstants.GREEN][j][i] = result[KLTConstants.GREEN][k];
             resultMatrix[KLTConstants.BLUE][j][i] = result[KLTConstants.BLUE][k++];
          }
       }

       return resultMatrix;
    }

    /** Convierte una imagen del espacio RGB a YCrCb
     * @param imageMatrix matriz de la imagen a convertir (en RGB)
     * @return matriz YCrCb
     */
    public static double [][][] convertToYCrCb(double [][][] imageMatrix)
    {
       return convertColorSpace(new Matrix(RGBToYCrCbMatrix), imageMatrix);
    }

    /** Convierte una imagen del espacio YCrCb a RGB
     * @param imageMatrix matriz de la imagen a convertir (en YCrCb)
     * @return matriz RGB
     */
    public static double [][][] convertToRGB(double [][][] imageMatrix)
    {
       return convertColorSpace(new Matrix(YCrCbToRGBMatrix), imageMatrix);
//       return convertColorSpace(new Matrix(RGBToYCrCbMatrix).inverse(), imageMatrix);
    }
}