package com.brownsoft.codec;
import java.io.*;

/**
 * <p>Title: Proyecto Codificacion de Imagenes y Video</p>
 * <p>Description: </p>
 * <p>Copyright: Copyright (c) 2004</p>
 * <p>Company: </p>
 * Esta clase implementa un bitstream codificado por huffman
 * Se definen 'canales' cada uno de los cuales puede tener su propia
 * tabla de Huffman.
 * Existe un canal especial 'UNENCODED' para el cual no se aplic la
 * codificacin de huffman
 * <p>Copyright: Copyright (c) 2004</p>
 * <p>Company: </p>
 * @author Gustavo Brown (alegus@adinet.com.uy)
 * @version 1.0
 */

public class HuffmanMultiplexerBitInputStream
{
   public static boolean DEBUG = false;

   private int curByte = 0;
   private int nextCurByte, nextNextCurByte;
   private byte curPos = 0;

   private static final int UNENCODED = HuffmanMultiplexerBitStream.UNENCODED;
   private static final int MAX_CHANNELS = HuffmanMultiplexerBitStream.MAX_CHANNELS;
   private static final double LOG_2 = Math.log(2);

   private FSMNode []ROOTs = new FSMNode[MAX_CHANNELS];
   private InputStream stream;
   private int elementsLeft;

   private static final StringBuffer NO_DEBUG = new StringBuffer("{Para ver el simbolo setear la constante DEBUG en true}");

//**********************************
// Intefaz pblica
//**********************************

   /** Construye una instancia a partir de un inputStream */
   public HuffmanMultiplexerBitInputStream(InputStream inputStream) throws IOException
   {
      this.stream = inputStream;

	  // Lo primero que hago es obtener la cantidad de total de elementos codificados
      // incluyendo los UNENCODED, representandolo en 4 bytes
      elementsLeft = (int)readBitStream(32);
      readDictionaries();
   }

   /** Lee cantBits del stream */
   public int readUnencoded(int cantBits) throws IOException
   {
      if(elementsLeft-- <= 0)
      {
         throw new KLTException("Se ha intentado leer ms simbolos de los que contiene el stream");
      }
      return (int)readBitStream(cantBits);
   }

   /** Decodifica un simbolo del stream */
   public int read(int channel) throws IOException
   {
      if(elementsLeft-- <= 0)
      {
         throw new KLTException("Se ha intentado leer ms simbolos de los que contiene el stream");
      }
      return decode(ROOTs[channel]);
   }


//**********************************
// Mtodos privados
//**********************************

    /** Ejecuta la maquina de estados para decodificar el prximo caracter
     *  @return el caracter decodificado
     **/
    private int decode(FSMNode ROOT)throws IOException
    {
       StringBuffer decodedString;
       if(DEBUG)
       {
	       	decodedString = new StringBuffer(); // Aqui almaceno el string decodificado para debug;
       }
       else
       {
          decodedString = NO_DEBUG;
       }
       FSMNode node = ROOT; // Iniciamos la maquina de estados en el ROOT
       // Hacemos la implementacion de la decodificacin iterativa en vez de recursiva
       // para obtener mejor performance
       do
       {
          boolean bit = readBit();
          if(DEBUG)
          {
	          decodedString.append(bit ? "1":"0");
          }
          // Primero chequeo que exista camino para ese lado
          if (!node.exists(bit))
          {
             throw new IOException("No se pudo decodificar la entrada '" + decodedString.toString() + "'");
          }
          node = node.get(bit);
       }while(!node.isValid());

       // Si pude decodificar el simbolo, lo retorno
       return node.getSymbol();
    }


   /** Lee los diccionarios asociados a esta instancia */
   private void readDictionaries()throws IOException
   {
		for(int i = 0; i < MAX_CHANNELS; i++)
        {
           ROOTs[i] = new FSMNode();
           if(readBit())
           { // Si el canal i fue utilizado
           		readDictionary(ROOTs[i]);
           }
        }
   }

   private void readDictionary(FSMNode ROOT)throws IOException
   {
      // El formato del diccionario es el siguiente:
      // 16 bits para indicar la cantidad de smbolos
      // Con la cantidad de smbolos se obtiene el largo mximo que puede tener cada palabra
      // que se calcula como log2(cant_simbolos) [+1 si no es entero]. Llamemos a ese numero max_bits_per_simbol
      // Luego por cada simbolo,
      //	- 2 bits para indicar con cuantos bytes voy a representar el simbolo
      //			(00= 1 byte, 01= 2 bytes, 10= 3 bytes, 11 = 4 bytes)
      //   - N bytes para indicar el smbolo segun lo descripto en el campo anterior
      //   - max_bits_per_symbol bits para indicar el largo del codeword
      //   - el codeword

      int cantSymbols = (int)readBitStream(16);
      int maxBitsPerSymbol = getCantBitsNeeded(cantSymbols);

      for (int i = 0; i < cantSymbols; i++)
      {
         int cantBytesThisSymbol = (int)readBitStream(2) + 1;
         int symbol = (int)readBitStream(8*cantBytesThisSymbol);
         int size = (int)readBitStream(maxBitsPerSymbol);
         addFsmNode(ROOT, symbol, size);
      }
   }

   /** Agrega un nodo a la maquina de estados finitos
    * @param node el nodeo actual dentro de la FSM
    * @param left cantidad de bits restantes del codeword
    **/
   private void addFsmNode(FSMNode node, int symbol, int left)throws IOException
   {
      if(left == 1)
      {
      	node.set(readBit(), new FSMNode(symbol));
      }
      else
      {
         addFsmNode(node.get(readBit()), symbol, left - 1);
      }
   }

    private int getCantBitsNeeded(int size)
    {
		return (int)Math.ceil(Math.log(size + 1) / LOG_2);
    }

   /** Lee un bit del stream
    * @return boolean indiciando el valor del bit ledo
    */
   private boolean readBit()throws IOException
   {
      if(curPos++ == 0)
      {
	      curByte = stream.read();
      }
      curPos %= 8;
      boolean bit = ((curByte & 1) == 1);
      curByte >>= 1;
      return bit;
   }

   /** Lee size bits del bitStream
    * @param size Cantidad de bits a leer
    */
   private long readBitStream(int size)throws IOException
   {
      long retValue = 0;

      for (int i = 0; i < size; i++)
      {
         retValue |= ((readBit() ? 1 : 0) << i);
      }
      return retValue;
   }

   class FSMNode
   {
      private boolean isValid = false;
      private int symbol;
      private FSMNode node0 = null;
      private FSMNode node1 = null;

      public FSMNode()
      {
         ;
      }

      public FSMNode(int symbol)
      {
         isValid = true;
         this.symbol = symbol;
      }

      public FSMNode get(boolean bit)
      {
         if (bit)
         {
            return get1();
         }
         else
         {
            return get0();
         }
      }

      public FSMNode get0()
      {
         if (node0 == null)
         {
            node0 = new FSMNode();
         }
         return node0;
      }

      public FSMNode get1()
      {
         if (node1 == null)
         {
            node1 = new FSMNode();
         }
         return node1;
      }

      public void set(boolean bit, FSMNode node)
      {
         if (bit)
         {
            set1(node);
         }
         else
         {
            set0(node);
         }
      }

      public void set0(FSMNode node)
      {
         if (node0 != null)
         {
            throw new IllegalArgumentException("Error al crear la maquina de estados finitos");
         }
         node0 = node;
      }

      public void set1(FSMNode node)
      {
         if (node1 != null)
         {
            throw new IllegalArgumentException("Error al crear la maquina de estados finitos");
         }
         node1 = node;
      }

      public boolean isValid()
      {
         return isValid;
      }

      public int getSymbol()
      {
         return symbol;
      }

      /** Indica si existe el subnodo */
      public boolean exists(boolean bit)
      {
         if (bit)
         {
            return node1 != null;
         }
         else
         {
            return node0 != null;
         }
      }
   }
}