package com.brownsoft.codec;
import java.util.*;
import java.io.*;

/**
 * <p>Title: Proyecto Codificacion de Imagenes y Video</p>
 * <p>Description: </p>
 * Esta clase implementa un bitstream codificado por huffman
 * Se definen 'canales' cada uno de los cuales puede tener su propia
 * tabla de Huffman. Las tablas de Huffman se van creando a medida que
 * se escriben datos por el canal y la codificacin se realiza cuando
 * se pide el bytearray de los datos.
 * Existe un canal especial 'UNENCODED' para el cual no se aplica la
 * codificacin de huffman (ni se calcula su tabla)
 * <p>Copyright: Copyright (c) 2004</p>
 * <p>Company: </p>
 * @author Gustavo Brown (alegus@adinet.com.uy)
 * @version 1.0
 */

public class HuffmanMultiplexerBitStream
{
   private int [] channelWrites;
   private Vector streamData;
   private HuffmanEncoder [] encoders;

//**********************************
// Intefaz pblica
//**********************************

   public static final int UNENCODED = -Integer.MAX_VALUE;
   public static final int MAX_CHANNELS = 5;

   public HuffmanMultiplexerBitStream()
   {
      streamData = new Vector();
      channelWrites = new int[MAX_CHANNELS];
   }

   /** Codifica el stream
    * @param stream el ByteArrayOutputStream donde codificar el stream
    * @param embedDictionary indica si embeber el diccionario
    * */
   public void encodeIntoByteArray(ByteArrayOutputStream stream)
   {
      if(encoders == null)
      {
         setDictionaries();
      }
      this.stream = stream;
      curByte = 0;
      curPos = 0;

      // Lo primero que hago es indicar la cantidad de total de elementos codificados
      // incluyendo los UNENCODED, representandolo en 4 bytes
      writeBitStream(streamData.size(), 32);
      // Ahora debo codificar los diccionarios
      writeDictionaries();

      // Ahora si codifico los elementos
      for(Enumeration enum = streamData.elements(); enum.hasMoreElements();)
      {
         StreamData streamData = (StreamData)enum.nextElement();
         int channel = streamData.channel;
         if(channel == UNENCODED)
         {
            writeBitStream(streamData.data, streamData.size);
         }
         else
         {
            HuffmanNode node = encoders[channel].encode(streamData.data);
            writeBitStream(node.getNode(), node.getSize());
         }
      }
      if(curPos != 0)
      {
         stream.write(curByte);
      }
   }

   /** Obtiene la cantidad de objetos escritos en esta instancia
    * @return cantidad de objetos escritos en esta instancia
    */
   public int getCantObjectsWritten()
   {
      return streamData.size();
   }

   /** Escribe size bits del parametro data en el bitStream
    * @param data El dato a escribir (los size bits menos significativos)
    * @param size Cantidad de bits a escribir
    */
   public void writeUnencodedBitStream(int data, int size)
   {
      streamData.addElement(new StreamData(UNENCODED, data, size));
   }

   /** Escribe size bits del parametro data en el bitStream
    * @param channel El canal sobre el cual escribir el dato
    * @param data El dato a escribir
    */
   public void write(int channel, int data)
   {
      streamData.addElement(new StreamData(channel, data));
      channelWrites[channel]++;
   }

   /** setea los diccionarios a partir de los datos de cada canal */
   public void setDictionaries()
   {
      encoders = new HuffmanEncoder[MAX_CHANNELS];
      int [][] datas = new int[MAX_CHANNELS][];
      for(int i = 0; i < MAX_CHANNELS; i++)
      {
         datas[i] = new int[channelWrites[i]];
         channelWrites[i] = 0;
      }

      // Obtengo los arrays de elementos de cada canal
      for(Enumeration enum = streamData.elements(); enum.hasMoreElements();)
      {
         StreamData streamData = (StreamData)enum.nextElement();
         int channel = streamData.channel;
         if(channel != UNENCODED)
         {
            datas[channel][channelWrites[channel]++] = streamData.data;
         }
      }

      // Ahora creo los encoders
      for(int i = 0; i < MAX_CHANNELS; i++)
      {
         encoders[i] = new HuffmanEncoder(datas[i]);
      }
   }

   /** Imprime estadisticas sobre esta instancia en pantalla */
   public void printStatics()
   {
      if(encoders == null)
      {
         setDictionaries();
      }
      for (int i = 0; i < MAX_CHANNELS; i++)
      {
         if (encoders[i].isValid())
         {
            System.err.println();
            System.err.println("Fuente " + i);
            encoders[i].printDictionary();
            System.err.println("Entropia de la fuente " + encoders[i].getSourceEntropy());
         }
      }

      System.err.println();
      System.err.println("Cantidad de objetos embebidos: " + getCantObjectsWritten());
   }

//**********************************
// Mtodos privados
//**********************************

   class StreamData
   {
      public int channel;
      public int data;
      public byte size;

      public StreamData(int channel, int data, int size)
      {
         this.channel = channel;
         this.data = data;
         this.size = (byte)size;
      }

      public StreamData(int channel, int data)
      {
         this.channel = channel;
         this.data = data;
         this.size = -1;
      }
   }

	private ByteArrayOutputStream stream;
    private byte curByte = 0;
	private byte curPos = 0;
    private static final double LOG_2 = Math.log(2);

    private int getCantBitsNeeded(int size)
    {
		return (int)Math.ceil(Math.log(size + 1) / LOG_2);
    }

    /** Escribe size bits del parametro data en el bitStream
     * @param data El dato a escribir (los size bits menos significativos)
     * @param size Cantidad de bits a escribir
     */
    private void writeBitStream(long data, int size)
    {
       for (int i = 0; i < size; i++)
       {
          curByte |= (data & 1) << curPos;
          data >>= 1;
          if (++curPos == 8)
          {
             stream.write(curByte);
             curByte = curPos = (byte)0;
          }
       }
    }

    /** Escribe size bits del parametro data[] en el bitStream
     * @param data El dato a escribir (los size bits menos significativos)
     * @param size Cantidad de bits a escribir
     */
    private void writeBitStream(long[] data, int size)
    {
       for (int i = 0; i < data.length; i++)
       {
          writeBitStream(data[i], size > 64 ? 64 : size);
          size -= 64;
       }
    }

	/** Finaliza la escritura del stream */
    private void finishStream()
    {
       if (curPos != 0)
       {
          stream.write(curByte);
       }
       stream = null;
    }

	/** Escribe todos los diccionarios asociados */
    private void writeDictionaries()
    {
       // Para cada canal voy a indicar con 1 bit
       // Si existe diccionario asociado a ese canal
       // y luego, si existe, escribo el diccionario
       for (int i = 0; i < MAX_CHANNELS; i++)
       {
          if (encoders[i].isValid())
          {
          	 writeBitStream(1, 1);
             writeDictionary(encoders[i]);
          }
          else
          {
          	 writeBitStream(0, 1);
          }
       }
     }

    /** Escribe el diccionario asociado al encoder en el bitStream
     * @param encoder El encoder cuyo diccionario se desea escribir
     */
    private void writeDictionary(HuffmanEncoder encoder)
    {
       // El formato del diccionario es el siguiente:
       // 16 bits para indicar la cantidad de smbolos
      // Con la cantidad de smbolos se obtiene el largo mximo que puede tener cada palabra
      // que se calcula como log2(cant_simbolos) [+1 si no es entero]. Llamemos a ese numero max_bits_per_simbol
       // Luego por cada simbolo,
       //	- 2 bits para indicar con cuantos bytes voy a representar el simbolo
       //			(00= 1 byte, 01= 2 bytes, 10= 3 bytes, 11 = 4 bytes)
       //   - N bytes para indicar el smbolo segun lo descripto en el campo anterior
       //   - max_bits_per_symbol bits para indicar el largo del codeword
       //   - el codeword
       short cantSymbols = (short)encoder.getCantSymbols();
       if (cantSymbols > 65536)
       {
          throw new KLTException("Puede haber como mximo 65536 simbolos");
       }

       // Guardo la cantidad de simbolos
       writeBitStream(cantSymbols, 16);
       int maxBitsPerSymbol = getCantBitsNeeded(cantSymbols);

       if (maxBitsPerSymbol == 0)
       {
          maxBitsPerSymbol++;
       }

       HuffmanNode[] symbols = encoder.getSymbols();
       for (int i = 0; i < symbols.length; i++)
       {
          int symbol = symbols[i].getSymbol();
          int N = 0;
          if(symbol > 0)
          {
          	N = getCantBitsNeeded(symbol)/8;
          }
          writeBitStream(N, 2);
          writeBitStream(symbols[i].getSymbol(), (N+1)*8);
          writeBitStream(symbols[i].getSize(), maxBitsPerSymbol);
          writeBitStream(symbols[i].getNode(), symbols[i].getSize());
       }
    }
 }