package com.brownsoft.codec;
import Jama.*;
import java.awt.*;
import java.awt.image.*;
import com.brownsoft.image.*;
import java.io.*;

/**
 * <p>Title: Proyecto Codificacion de Imagenes y Video</p>
 * <p>Description: </p>
 * Esta clase implementa un codificador de imgenes a color y B/N aplicando la
 * transformacin de Karhunen-Love
 * <p>Copyright: Copyright (c) 2004</p>
 * <p>Company: </p>
 * @author Gustavo Brown (alegus@adinet.com.uy)
 * @version 1.0
 */

public class KLTEncoder implements KLTConstants
{
   private Image image;
   private int quality;
   private int coefficients;
   private boolean embedKLTVectors;
   private HuffmanMultiplexerBitStream huffmanMultiplexer;
   private HuffmanMultiplexerBitStream externalKLTWriter;
   private HuffmanMultiplexerBitInputStream externalKLTReader;
   private ByteArrayOutputStream stream;
   private DataOutputStream dataStream;
   private String comment = "";
   private byte imageType;
   private byte outputImageType;
   private int imageWidth, imageHeight;
   private int []imageVector;
   private int cantBands;
   private int currentBand;
   private double[][][] imageMatrix;
   private double[][][] imageMatrixYCrCb;
   private int currentBandLeastUsedCoeff;
   private boolean showBands = false;
   private boolean showProgress;
   private String externalKLTBaseOutputName;
   private String externalKLTBaseInputName;
   private boolean useExternalKLTBase = false;
   private boolean writeExternalKLTBase = false;

   protected int BLOCK_WIDTH;
   protected int BLOCK_SIZE;

   public static final boolean DEBUG = true;

//**********************************
// Intefaz pblica
//**********************************

   /** Crea una instancia de KLTEncoder
    * @param image La imagen a codificar
    */
   public KLTEncoder(Image image)
   {
      try
      {
         this.image = image;
         setImageMatrix();
         setBlockSize(8);
         setQuality(75);
         setEmbedKLTVectors(true);
         setOutputImageType(IMAGE_DEFAULT);
      } catch (KLTException e)
      {
         e.printStackTrace();
      }
   }

	/** Setea el comentario de la imagen */
	public void setComment(String comment)
    {
       this.comment = comment;
    }

	/** Obtiene el comentario de la imagen */
    public String getComment()
    {
       return comment;
    }

	/** Setea la calidad de la compresin */
    public void setQuality(int quality)throws KLTException
    {
      if(quality < 0 || quality > 100)
      {
         throw new KLTException("La calidad debe ser entre 0 y 100");
      }
      this.quality = quality;
    }

	/** Obtiene la calidad de compresion de la imagen */
    public int getQuality()
    {
       return quality;
    }

    /** Setea la cantidad de coeficientes de la base a utilizar
     * @param coefficients Cantidad de coeficientes a utilizar
     */
    public void setNumberOfCoefficients(int coefficients) throws KLTException
    {
		if(coefficients < 1 || coefficients > BLOCK_SIZE)
        {
           throw new KLTException("La cantidad de coeficientes debe valer entre 1 y " + BLOCK_SIZE);
        }
        this.coefficients = coefficients;
    }

    /** Retorna la cantidad de coeficientes de la base a utilizar
     *  @return cantidad de coeficientes de la base a utilizar
     */
    public int getNumberOfCoefficients()
    {
       return coefficients;
    }

	/** Indica si embeber los vectores de la base de la KLT */
    public void setEmbedKLTVectors(boolean embedKLTVectors)
    {
       this.embedKLTVectors = embedKLTVectors;
    }

	/** Retorna si embeber los vectores de la base de la KLT */
    public boolean getEmbedKLTVectors()
    {
       return embedKLTVectors;
    }

	/** Setea el tamao(lado) de los bloques */
    public void setBlockSize(int blockSize) throws KLTException
    {
       if(blockSize < 1 || blockSize > 255)
       {
          throw new KLTException("El tamao del bloque debe estar entre 1 y 255");
       }
       this.BLOCK_WIDTH = blockSize;
       this.BLOCK_SIZE = blockSize * blockSize;
    }

	/** Obtiene el tamao(lado) de los bloques */
    public int getBlockSize()
    {
       return BLOCK_WIDTH;
    }

	/** Indica si se desea mostrar las bandas que se van codificando */
    public void setShowBands(boolean showBands)
    {
       this.showBands = showBands;
    }

	/** Retorna si se mostraran las bandas al codificar la imagen */
    public boolean getShowBands()
    {
       return showBands;
    }

   /** Codifica la imagen retornando un array de bytes
    */
   public byte [] encodeImage()throws KLTException, IOException
   {
      return encode();
   }

   /** Setea si se desea ir mostrando un progreso de la codificacin
    * @param showProgress boolean que indica si mostrar progreso
   **/
   public void setShowProgress(boolean showProgress)
   {
      this.showProgress = showProgress;
   }

   /** Indica si se va a mostrar progreso de la codificacion
    */
   public boolean getShowProgress()
   {
      return showProgress;
   }

   /** Setea el tipo de la imagen de salida
    * @param outputImageType tipo de la imagen de salida (IMAGE_DEFAULT, IMAGE_GRAY, IMAGE_RGB o IMAGE_YCrCb)
    */
   public void setOutputImageType(byte outputImageType)throws KLTException
   {
      switch(outputImageType)
      {
         case IMAGE_RGB:
         case IMAGE_YCrCb:
         	if(imageType == IMAGE_GRAY)
            { // Los formatos RGB y YCrCb solo se soportan para imagenes a colores
               return;
            }
         case IMAGE_GRAY:
         	this.outputImageType = outputImageType;
             break;
         case IMAGE_DEFAULT:
			this.outputImageType = imageType;
             break;
         default:
         	throw new KLTException("Tipo de imagen de salida no soportado: " + outputImageType);
      }
      if(this.outputImageType == IMAGE_GRAY)
      {
         cantBands = 1;
         if(imageType != IMAGE_GRAY)
         {
            setImageMatrixYCrCb();
         }
      }
      else
      {
         cantBands = 3;
         if(outputImageType == IMAGE_YCrCb)
         {
            if (imageType != IMAGE_GRAY)
            {
               setImageMatrixYCrCb();
            }
         }
      }
   }

   /** Indica el tipo de imagen de salida
    * @return Una de las constantes IMAGE_DEFAULT, IMAGE_GRAY, IMAGE_RGB o IMAGE_YCrCb
    */
   public byte getOutputImageType()
   {
      return outputImageType;
   }

   /** Indica que utilice la base de la klt a partir del archivo pasado por argumento
    * @param externalKLTBase Nombre del archivo donde obtener la base de la KLT
    */
   public void setKLTBaseName(String kltBaseName)
   {
      useExternalKLTBase = true;
      externalKLTBaseInputName = kltBaseName;
   }

   /** Obtiene el nombre del archivo de donde obtener la base de la KLT o null si
    * debe ser calculado
    * @return Nombre del archivo de donde obtener la base de la KLT o null si la misma debe ser calculada
    */
   public String getKLTBaseInputName()
   {
      return externalKLTBaseInputName;
   }

   /** Setea si se desea crear la base de la KLT en un archivo separado
    * @param externalKLTBase Nombre del archivo donde almacenar la base de la KLT
    * @see setEmbedKLTVectors
    */
   public void setKLTBaseOutputName(String externalKLTBase)
   {
      externalKLTBaseOutputName = externalKLTBase;
      writeExternalKLTBase = (externalKLTBase != null);
   }

   /** Obtiene el nombre del archivo donde almacenar la base de la KLT
    * @return Nombre del archivo donde se almacenara la base de la KLT
    * @see setEmbedKLTVectors
    */
   public String getKLTBaseOutputName()
   {
      return externalKLTBaseOutputName;
   }

   /** Codifica la imagen y escribe el archivo de salida
    */
   public void encodeImage(String outputFileName)throws KLTException, IOException
   {
      byte [] buffer = encodeImage();
      BufferedOutputStream outputStream = new BufferedOutputStream(new FileOutputStream(outputFileName));
      outputStream.write(buffer);
      outputStream.close();
   }


//**********************************
// Mtodos del package
//**********************************

	/** Obtiene la matriz de un canal de la imagen
     * @param band Canal a obtener
     * @return double[][] con la matriz de ese canal
     **/
	protected double[][] getImageMatrix(int band)
    {
	   if(band >= cantBands)
       {
          throw new KLTException("No existe la banda " + band);
       }
       switch(outputImageType)
       {
          case IMAGE_GRAY:
          	if(imageType == IMAGE_RGB)
            {
               return imageMatrixYCrCb[band];
            }
            else
            {
             return imageMatrix[band];
            }
          case IMAGE_RGB:
             return imageMatrix[band];
          case IMAGE_YCrCb:
               return imageMatrixYCrCb[band];
          default:
          {
             throw new KLTException("Todavia no esta implementado");
          }
       }
    }

//**********************************
// Mtodos privados
//**********************************

   /** Codifica la imagen y retorna un bytearray con el contenido */
   private byte [] encode()throws IOException
   {
		stream = new ByteArrayOutputStream();
        dataStream = new DataOutputStream(stream);
        huffmanMultiplexer = new HuffmanMultiplexerBitStream();
        if(writeExternalKLTBase)
        { // Si voy a guardar la base de la klt en un archivo aparte
           externalKLTWriter = new HuffmanMultiplexerBitStream();
        }
        if(useExternalKLTBase)
        {
        	openExternalKLTBase();
	    }
        writeHeader();
        compressImage();
		return stream.toByteArray();
   }

   /** Escribe el header */
   private void writeHeader()throws IOException
   {
      // Escribe el Header de la imagen
      // El header esta compuesto:
      // FORMAT_STRING -> String (34 bytes) que identifica el formato de la imagen
      // COMMENT_SIZE -> Short (2 bytes) que indican la cantidad de bytes del comentario de la imagen
      // FLAGS -> 1 byte con las flags de la imagen
      //        \-> Bit 0: Indica si los vectores de la KLT estan embebidos en la imagen
      //        \-> Bit 1-2: Indica el formato de la imagen
      //         \--> 00: Imagen en escala de grises
      //              10: Imagen RGB
      //              11: Imagen YCrCb
      // BLOCK_SIZE -> 1 byte indicando el tamao(lado) de cada bloque (bloque N*N)
      // QUALITY -> 1 byte indicando la calidad de la imagen
      // IMAGE_WIDTH -> 2 bytes indicando el ancho de la imagen
      // IMAGE_HEIGHT -> 2 bytes indicando el alto de la imagen

      writeString(FORMAT_STRING);
      dataStream.writeShort(comment.length());
      if(comment.length() > 0)
      {
         writeString(comment);
      }
      dataStream.writeByte(getFlags());
      dataStream.write(BLOCK_WIDTH);
      dataStream.writeByte(quality);
      dataStream.writeShort(imageWidth);
      dataStream.writeShort(imageHeight);
   }

   /** Aplica la KLT a la imagen con el factor de calidad deseado y obtiene los coeficientes
   *   tanto de la base como de la propia transformacin
   */
   private void compressImage() throws IOException
   {
      // Ahora debo aplicar la KLT a cada banda de la imagen
      for(currentBand = 0; currentBand < cantBands; currentBand++)
      {
         KLTHelper helper;
         if(useExternalKLTBase)
         { // Si debo leer la base de la KLT en desde el externalKLTReader

            currentBandLeastUsedCoeff = externalKLTReader.readUnencoded(16);
            // Decodifico uno a uno los versores utilizados
            double [][] invBasis = new double[BLOCK_SIZE][BLOCK_SIZE];
            for (int i = BLOCK_SIZE - 1; i >= currentBandLeastUsedCoeff; i--)
            {
               for (int j = 0; j < BLOCK_SIZE; j++)
               {
                  int cantBitsNeeded = externalKLTReader.read(DC_CHANNEL);
                  invBasis[i][j] = getDoubleFromVLI(externalKLTReader, cantBitsNeeded) / KLT_COEFF_MULTIPLIER;
               }
            }
            helper = new KLTHelper(BLOCK_WIDTH, BLOCK_SIZE, getImageMatrix(currentBand), invBasis);
         }
         else
         { // Si debo calcular la base de la KLT
	         helper = new KLTHelper(BLOCK_WIDTH, BLOCK_SIZE, getImageMatrix(currentBand));
         }
         currentBandLeastUsedCoeff = BLOCK_SIZE - 1;

         // Obtengo los vectores con los coeficientes de la imagen
         double[][] decodedImageMatrixCoeffs = helper.applyKLTByNumberOfCoefficients(getNumberOfCoefficients());

		if(showBands)
        {
           // Ahora instancio un PGMDecoder con estos datos
           int imageWidth = this.imageWidth;
           int imageHeight = this.imageHeight;
           if(outputImageType == IMAGE_YCrCb && currentBand != AXIS_Y)
           {
              imageWidth /= subsampling_x;
              imageHeight /= subsampling_y;
           }
           PGMEncoder decodedImage = new PGMEncoder(helper.unblockRearrange(new Matrix(decodedImageMatrixCoeffs).times(helper.invEigenVectors).getArrayCopy(), imageWidth, imageHeight, BLOCK_WIDTH));
           decodedImage.showImage(getBandName(currentBand));
        }

         // Ahora codifico el bloque
         double lastDC = 0;
         for(int i = 0; i < decodedImageMatrixCoeffs.length; i++)
         {
            if(showProgress)
            {
               System.err.print("\rProgreso: " + (((currentBand * decodedImageMatrixCoeffs.length + i+1)*100)/(decodedImageMatrixCoeffs.length * cantBands)) + "%\t\t");
            }
            lastDC = encodeBlock(decodedImageMatrixCoeffs[i], lastDC);
         }

         // Veo si debo incluir la base de la KLT
         if (showProgress)
         {
            System.err.println("\nLeastUsedCoeff=" + currentBandLeastUsedCoeff + "  ");
         }

         if (embedKLTVectors)
         {
            // Indico la cantidad de coeficientes que utilic en esta banda
            huffmanMultiplexer.writeUnencodedBitStream(currentBandLeastUsedCoeff, 16);
            // Si debo incluir la base de la KLT en el archivo, solo codifico los versores utilizados
            // Ahora voy codificando uno a uno los versores utilizados
            double[][] invBasis = helper.getInvBasis();
            for (int i = BLOCK_SIZE - 1; i >= currentBandLeastUsedCoeff; i--)
            {
               for (int j = 0; j < BLOCK_SIZE; j++)
               {
                  double value = invBasis[i][j] * KLT_COEFF_MULTIPLIER;
                  int cantBitsNeeded = getCantBitsNeeded(value);
                  huffmanMultiplexer.write(DC_CHANNEL, cantBitsNeeded); // Codifico el size del coef dc
                  huffmanMultiplexer.writeUnencodedBitStream(getVLI(cantBitsNeeded, value), cantBitsNeeded);
               }
            }
         }

         if(writeExternalKLTBase)
         {  // Si debo guardar la base de la KLT en un archivo separado codifico todos los versores
            externalKLTWriter.writeUnencodedBitStream(0, 16); // Indico que se utilizaron TODOS los versores

            // Y ahora voy codificando uno a uno los versores utilizados
            double[][] invBasis = helper.getInvBasis();
            for (int i = BLOCK_SIZE - 1; i >= 0; i--)
            {
               for (int j = 0; j < BLOCK_SIZE; j++)
               {
                  double value = invBasis[i][j] * KLT_COEFF_MULTIPLIER;
                  int cantBitsNeeded = getCantBitsNeeded(value);
                  externalKLTWriter.write(DC_CHANNEL, cantBitsNeeded); // Codifico el size del coef dc
                  externalKLTWriter.writeUnencodedBitStream(getVLI(cantBitsNeeded, value), cantBitsNeeded);
               }
            }
         }
      }

      // Ahora si puedo armar la codificacin huffman de los canales codificados
      huffmanMultiplexer.encodeIntoByteArray(stream);

      if(showProgress)
      {
		huffmanMultiplexer.printStatics();
	  }

      if (writeExternalKLTBase)
      { // Si debo guardar la base de la KLT en un archivo separado
      	try
        {
      		FileOutputStream kltOutputStream = new FileOutputStream(externalKLTBaseOutputName);
            // Primero guardo el header del KLTC
            kltOutputStream.write("kltc".getBytes());
            // Ahora almaceno el BLOCK_WIDTH
            kltOutputStream.write(BLOCK_WIDTH);
            // Ahora si almaceno la base de la klt
            ByteArrayOutputStream tempArray = new ByteArrayOutputStream();
            externalKLTWriter.encodeIntoByteArray(tempArray);
            tempArray.close();
            kltOutputStream.write(tempArray.toByteArray());
            kltOutputStream.close();
        }catch(IOException e)
        {
           System.err.println("Warning: No se pudo guardar la base de la KLT en el archivo '" + externalKLTBaseOutputName + "'");
           System.err.println(e.toString());
        }
      }
   }

   /** Codifica un bloque de la imagen
    * @param vector Vector de coeficientes del bloque
    * @param lastDC el coeficiente de continua del bloque anterior
    * @return double El coeficiente de continua actual
    */
   private double encodeBlock(double[] vector, double lastDC)
   {
      // Lo primero que debo hacer es codificar el coeficiente de contnua
      double currentDC = quantizeDC(vector[vector.length - 1]);
      int cantBitsNeeded = getCantBitsNeeded(currentDC - lastDC);
      huffmanMultiplexer.write(DC_CHANNEL, cantBitsNeeded); // Codifico el size del coef dc
      huffmanMultiplexer.writeUnencodedBitStream(getVLI(cantBitsNeeded, currentDC - lastDC), cantBitsNeeded);

      // Ahora comienzo de atras para adelante a codificar los coeficientes de alterna
      // Para la KLT se cumple que los versores estan en orden creciente de importancia

      int curCoeff = vector.length - 2;
      int cantZeros;
      while(true)
      {
		// Ahora debo contar la cantidad de ceros seguidas que quedan a partir de aqui
         for(cantZeros = 0; curCoeff >= 0; cantZeros++)
         {
            if(!isZero(quantizeAC(vector[curCoeff], curCoeff)))
            {
               break;
            }
            curCoeff--;
         }

         if(curCoeff == -1)
         { // Si era el ltimo coeficiente, termino
         	huffmanMultiplexer.write(AC_CHANNEL_BASE + currentBand, EOB);
            break;
         }

         if(curCoeff < currentBandLeastUsedCoeff)
         { // Indico hasta que coeficiente embeber en el archivo
            currentBandLeastUsedCoeff = curCoeff;
         }

         int zeroRunLength = cantZeros << 4; // Dejo los 4 bits menos significativos para indicar la cantidad de bits
         double coeffValue = quantizeAC(vector[curCoeff], curCoeff);
         cantBitsNeeded = getCantBitsNeeded(coeffValue);
         if(cantBitsNeeded > 15)
         { // No debera pasar...
         	throw new KLTException("Un coeficiente de alterna puede necesitar a lo sumo 15 bits para ser codificado");
//         	cantBitsNeeded = 15;
         }
         huffmanMultiplexer.write(AC_CHANNEL_BASE + currentBand, (zeroRunLength|cantBitsNeeded));
         huffmanMultiplexer.writeUnencodedBitStream(getVLI(cantBitsNeeded, coeffValue), cantBitsNeeded);

		 curCoeff--;
         if(curCoeff == -1)
         { // Si era el ltimo coeficiente, termino
         	huffmanMultiplexer.write(AC_CHANNEL_BASE + currentBand, EOB);
            break;
         }
      }
      return currentDC;
   }

   /** Retorna true si el valor pasado por argumento se considera nulo */
   private boolean isZero(double value)
   {
      return Math.abs(value * FRACTION_MULTIPLIER) <= 1.5;
   }

   /** Retorna la cantidad de bits que se necesita para expresar una magnitud */
   private int getCantBitsNeeded(double value)
   {
      if(value < 0)value = -value;
      int integerValue = (int)Math.round(value * FRACTION_MULTIPLIER);
      if(integerValue < 2)
      {
         integerValue = 2;
      }
      return (int)Math.ceil(Math.log(integerValue+1)/LOG_2);
   }

   /** Obtiene un VariableLengthInteger */
   private int getVLI(int cantBitsNeeded, double value)
   {
      int sign = 0;
      if(value < 0)
      {
         value = -value;
         sign = (1 << (cantBitsNeeded-1));
      }
      int integerValue = (int)Math.round(value * FRACTION_MULTIPLIER);
      return sign + integerValue - (1<<(cantBitsNeeded-1));
   }

   /** Obtiene las flags de la imagen */
   private byte getFlags()
   {
      int flags = (outputImageType & 3) << 1;
      flags |= embedKLTVectors ? (byte)1 : 0;
      return (byte) flags;
   }


   /** Escribe un String en el stream */
   private void writeString(String str)throws IOException
   {
      stream.write(str.getBytes());
   }

   /** Setea la informacion sobre la imagen de entrada y su matriz asociada */
   private void setImageMatrix()
   {
      if (image instanceof BufferedImage && ((BufferedImage)image).getType() == BufferedImage.TYPE_BYTE_GRAY)
      { // Si se trata de una imagen en escala de grises
         BufferedImage image = (BufferedImage)this.image;
         imageType = IMAGE_GRAY;
         imageWidth = image.getWidth();
         imageHeight = image.getHeight();
         imageMatrix = new double[1][imageHeight][imageWidth];
         Raster raster = image.getData();
         int[] vector = new int[imageHeight * imageWidth];
         raster.getPixels(0, 0, imageWidth, imageHeight, vector);
         int k = 0;
         for (int j = 0; j < imageHeight; j++)
         {
            for (int i = 0; i < imageWidth; i++)
            {
               int pixel = vector[k++];
               imageMatrix[GRAY][j][i] = (pixel) & 0xff;
            }
         }
      }
      else
      {
         Component observer = new Component(){};
         MediaTracker tracker = new MediaTracker(observer);
         tracker.addImage(image, 0);
         try
         {
            tracker.waitForID(0); // Espero hasta que la imagen este completamente disponible
         } catch (InterruptedException e)
         {
            ;
         }

         imageWidth = image.getWidth(observer);
         imageHeight = image.getHeight(observer);
         imageType = IMAGE_RGB;
         imageMatrix = new double[3][imageHeight][imageWidth];

		 int []vector = new int[imageWidth * imageHeight];
         PixelGrabber pixelGrabber = new PixelGrabber(image, 0, 0, imageWidth, imageHeight, vector, 0, imageWidth);
         try
         {
            pixelGrabber.grabPixels();
         } catch (InterruptedException e)
         {
            ;
         }
         int k = 0;
         for(int j = 0; j < imageHeight; j++)
         {
            for(int i = 0; i < imageWidth; i++)
            {
               int pixel = vector[k++];
               imageMatrix[RED][j][i] = (pixel >> 16) & 0xff;
               imageMatrix[GREEN][j][i] = (pixel >> 8) & 0xff;
               imageMatrix[BLUE][j][i] = (pixel) & 0xff;
            }
         }
       }
   }

   /** Calcula la matrix en el espacio YCrCb
    *
    */
   private void setImageMatrixYCrCb()
   {
      // Primero me cambio al especio YCrCb
      double [][][] tempImageYCrCb = KLTHelper.convertToYCrCb(imageMatrix);
      // Ahora subsampleo los canales Cr y Cb
      imageMatrixYCrCb = new double[3][][];
      imageMatrixYCrCb[0] = tempImageYCrCb[0]; // El canal Y no se subsamplea


	  int subsampledHeight = (imageHeight + subsampling_y-1)/subsampling_y;
	  int subsampledWidth = (imageWidth + subsampling_x-1)/subsampling_x;
 	  imageMatrixYCrCb[1] = new double[subsampledHeight][subsampledWidth];
 	  imageMatrixYCrCb[2] = new double[subsampledHeight][subsampledWidth];
      int curY = 0;
      for(int j = 0; j < imageHeight; j += subsampling_y)
      {
         int curX = 0;
         for(int i = 0; i < imageWidth; i += subsampling_x)
         {
            double mediumValueCr = 0;
            double mediumValueCb = 0;
            int cantValues = 0;
            for(int _j = 0; _j < subsampling_y; _j++)
            {
	            for(int _i = 0; _i < subsampling_x; _i++)
                {
                   try
                   {
	                   mediumValueCr += tempImageYCrCb[1][j+_j][i+_i];
	                   mediumValueCb += tempImageYCrCb[2][j+_j][i+_i];
                       cantValues++;
                   }catch(ArrayIndexOutOfBoundsException outOfBounds){ ; } // Solo considero los pixels por los cuales subsampleo
                }
            }
            imageMatrixYCrCb[1][curY][curX] = mediumValueCr/cantValues; // Seteo el promedio de los values
            imageMatrixYCrCb[2][curY][curX++] = mediumValueCb/cantValues;
         }
         curY++;
      }
   }

   /** Obtiene el nombre asociado a un canal
    * @param band Canal
    * @return Nombre asociado a un canal
    */
   private String getBandName(int band)
   {
      switch(outputImageType)
      {
         case IMAGE_GRAY: return "Canal Gris";
         case IMAGE_RGB: return "Canal " + (band == 0 ? "R" : band == 1 ? "G": "B");
         case IMAGE_YCrCb: return "Canal " + (band == 0 ? "Y" : band == 1 ? "Cr": "Cb");
		 default:  return "Desconocido";
      }
   }

   /** Cuantifica el componente de continua */
   private double quantizeDC(double value)
   {
      return value / DC_QUANTIZER;
   }

   /** Cuantifica un componente de alterna */
   private double quantizeAC(double value, int curCoef)
   {
      if(outputImageType == IMAGE_YCrCb && currentBand != AXIS_Y)
      { // Si estoy codificando en YCrCb las cromas las codifico ms grueso
	      return (value / AC_QUANTIZER_CrCb)*quality * Math.pow(AC_QUANTIZER_FACTOR, curCoef);
      }
      else
      {
	      return (value / AC_QUANTIZER)*quality * Math.pow(AC_QUANTIZER_FACTOR, curCoef);
      }
   }

   /** Decuantifica el componente de continua */
   private double dequantizeDC(double value)
   {
      return value * DC_QUANTIZER;
   }

   /** Decuantifica un componente de alterna */
   private double dequantizeAC(double value, int curCoef)
   {
      if(imageType == IMAGE_YCrCb && currentBand != AXIS_Y)
      { // Si estoy decodificando en YCrCb las cromas las codifico ms grueso
	      return (value *  AC_QUANTIZER_CrCb) / (Math.pow(AC_QUANTIZER_FACTOR, curCoef) * quality) ;
      }
      else
      {
	      return (value *  AC_QUANTIZER) / (Math.pow(AC_QUANTIZER_FACTOR, curCoef) * quality) ;
      }
   }

   /** Obtiene un double a partir de un VariableLengthInteger
    * @param huffmanDecoder el HuffmanMultiplexerBitInputStream con el stream de entrada
    * @param cantBitsNeeded cantidad de bits de este VLI
    * */
   private double getDoubleFromVLI(HuffmanMultiplexerBitInputStream huffmanDecoder, int cantBitsNeeded)throws IOException
   {
      int value = huffmanDecoder.readUnencoded(cantBitsNeeded);
      int cte = (1<<(cantBitsNeeded-1));
      boolean isNegative = ((value & cte) != 0);
      value &= ~cte;
      int sign = 0;

      value += cte;
      if(isNegative)
      {
         value = -value;
      }
      return (((double)value) / FRACTION_MULTIPLIER);
   }

	/** Abre un archivo de base de la KLT (.kltc) */
    private void openExternalKLTBase() throws IOException
    {
       BufferedInputStream kltInputStream = new BufferedInputStream(new FileInputStream(externalKLTBaseInputName));
       // Primero leo el String que identifica los streams de klt
       byte[] tempBuffer = new byte[4];
       kltInputStream.read(tempBuffer);
       if (!new String(tempBuffer).equals("kltc"))
       {
          throw new KLTException("Archivo de coeficientes de la base de la KLT invlido (" + externalKLTBaseInputName + ")");
       }
       // Ahora veo si el BLOCK_WIDTH es el mismo
       int embededBlockWidth = kltInputStream.read() & 255;
       if (embededBlockWidth != BLOCK_WIDTH)
       {
          System.err.println("Warning: El archivo de coeficientes difiere en el tamao del bloque");
          System.err.println("         Se utilizara como valor " + embededBlockWidth + " en vez de " + BLOCK_WIDTH);
          setBlockSize(embededBlockWidth);
       }
       externalKLTReader = new HuffmanMultiplexerBitInputStream(kltInputStream);
    }
}