package com.brownsoft.codec;
import java.math.BigDecimal;
import java.util.*;

/**
 * Esta clase es en conjunto con HuffmanOutputStream quien codifica un stream mediante Huffman
 *
 * <p>Title: Proyecto Codificacion de Imagenes y Video</p>
 * <p>Description: </p>
 * <p>Copyright: Copyright (c) 2003</p>
 * <p>Company: </p>
 * @author Gustavo Brown (alegus@adinet.com.uy)
 * @version 1.0
 */

public class HuffmanEncoder
{
   protected static final int MAX_SYMBOL_SIZE = 256;
   private Hashtable nodes;
   private int cantSymbols = 0;
   HuffmanNode [] symbols = null;

   /** Crea una instancia de huffman */
   public void HuffmanEncoder() { ; }

   /** Crea una instancia de huffman con un diccionario dado
    * El constructor recibe un array de int con el cual se determinan los simbolos de la fuente
    * y la distribucion de probabilidades
    * @param sourceDictionary diccionario a utilizar
    */
   public HuffmanEncoder(int [] sourceDictionary)
   {
      setSourceDictionary(sourceDictionary);
   }

   /** Crea el diccionario que va a utilizar el codificador
    * @param sourceDictionary array de int de donde sacar la distribucion de probabilidades
    */
   public void setSourceDictionary(int[] sourceDictionary)
   {
      int size = sourceDictionary.length;

      // Primero calculo las frecuencias
      Hashtable freqs = new Hashtable();

      for(int i = 0; i < sourceDictionary.length; i++)
      {
         Integer value = new Integer(sourceDictionary[i]);
         int [] thisFreq = (int[])freqs.get(value);
         if(thisFreq == null)
         {
		thisFreq = new int[]{1};
        	freqs.put(value, thisFreq);
         }
         else
         {
            thisFreq[0]++;
         }
      }

//--      nodes = new HuffmanNode[MAX_SYMBOL_SIZE];

      // Ahora calculo las probabilidades de cada simbolo y la cantidad total de smbolos
      // En nodes[i] voy a tener la codificacion del simbolo i
      nodes = new Hashtable();
      int diffSymbols = freqs.size();
      HuffmanNode[] tempProps = new HuffmanNode[diffSymbols];
      int j = 0;
      for(Enumeration enum = freqs.keys(); enum.hasMoreElements();j++)
      {
         Integer value = (Integer)enum.nextElement();
         int thisFreq = ((int[])freqs.get(value))[0];
         tempProps[j] = new HuffmanNode(value.intValue(), (double)thisFreq / size);
         nodes.put(value, tempProps[j]);
      }

      cantSymbols = diffSymbols;
      if(diffSymbols == 0)
      {
         return;
      }

      // Ahora genero el rbol
      for(int curSymbols = diffSymbols; curSymbols > 1; curSymbols--)
      {
         // Primero ordeno las probabilidades
         // A priori esto es O(n^2), pero el array
         // se va a mantener semiordenado
         sort(tempProps, curSymbols);
         tempProps[curSymbols - 2] = new HuffmanNode(tempProps[curSymbols - 2], tempProps[curSymbols - 1]);
      }

      // Ok, ahora a partir de tempProps[0] puedo armar el diccionario de cada smbolo
      // Esto se hace recursivamente dentro de HuffmanNode
      tempProps[0].setNode(new long [0], 0);

      symbols = new HuffmanNode[diffSymbols];

      j = 0;
      for(Enumeration enum = nodes.elements(); enum.hasMoreElements();j++)
      {
            symbols[j] = (HuffmanNode)enum.nextElement();
      }
   }

   /** Ordena de mayor a menor */
   private void sort(HuffmanNode [] nodes, int size)
   {
      // Aplico un bubble sort al array de nodes
      // Paro el algoritmo cuando no hay ningun swap
      boolean swapped = false;
      for(int i = size - 1; i > 0; i--)
      {
         for(int j = i ; j > 0; j--)
         {
            if (nodes[j].getProbability() > nodes[j - 1].getProbability())
            {
               HuffmanNode tempNode = nodes[j];
               nodes[j] = nodes[j-1];
               nodes[j-1] = tempNode;
               swapped = true;
            }
         }
         if(!swapped)
         {
            break;
         }
      }
   }

   /** Esta clase imprime el diccionario */
   public void printDictionary()
   {
      if(symbols == null)
      {
         return;
      }
      for(int i = 0; i < symbols.length; i++)
      {
         System.out.println(symbols[i].toString());
      }
   }

   /** Obtiene la cantidad de simbolos del diccionario
    * @return Cantidad de simbolos del diccionario
    */
   public int getCantSymbols()
   {
      return cantSymbols;
   }

   /** Retorna un array de HuffmanNode con el codeword del smbolo */
   public HuffmanNode [] getSymbols()
   {
      return symbols;
   }

   /** Obtiene el HuffmanNode asociado a un simbolo
    * @param symbol simbolo
    * @param HuffmanNode asociado a dicho simbolo
    */
   public HuffmanNode encode(int symbol)throws IllegalArgumentException
   {
      if(nodes == null)
      {
         throw new IllegalArgumentException("No se ha definido diccionario para esta instancia de HuffmanEncoder");
      }

      HuffmanNode encoder = (HuffmanNode)nodes.get(new Integer(symbol));
      if(encoder == null || !encoder.isValid())
      {
         throw new IllegalArgumentException("El diccionario no contiene una entrada para el smbolo 0x" + Integer.toHexString(symbol));
      }

      return encoder;
   }

   /** Calcula la entropa de la fuente
    * @return la entropa de la fuente
    */
   public double getSourceEntropy()
   {
      // Tabajo con BigDecimals para obtener mejor precisin, pues sino debera primero
      // calcular la entropa de los smbolos con menor probabilidad para mejor precisin
      BigDecimal entropia = new BigDecimal(0);
      BigDecimal log2 = new BigDecimal(Math.log(2));

      for(int i = 0; i < symbols.length; i++)
      {
         BigDecimal thisSymbol = new BigDecimal(symbols[i].getProbability());
         entropia = entropia.subtract(thisSymbol.multiply(new BigDecimal(Math.log(thisSymbol.doubleValue())).divide(log2, BigDecimal.ROUND_UP)));
      }
      return entropia.doubleValue();
   }

   public boolean isValid()
   {
      return symbols != null;
   }

   public int getCantBitsMaxCodeword()
   {
      int cantBitsMaxCodeword = 0;
      for(int i = 0; i < symbols.length;i++)
      {
         int cantBits = symbols[i].getSize();
         if(cantBits > cantBitsMaxCodeword)
         {
            cantBitsMaxCodeword = cantBits;
         }
      }
      return cantBitsMaxCodeword;
   }
}
