% Author: Gustavo Brown (gbrown@fing.edu.uy)
% Start predicate: main
% Compiles with SWI Prolog & Sicstus Prolog
% Note: To compile with Sicstus Prolog, search for the commented snippet
% "SICSTUS - compatibility pack" and uncomment it (until "Fin SICSTUS - compatibility pack")
% They contain an implementation of some predicates not found in Sicstus Prolog 
% Date: 22/06/2009

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Cmd line                          %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

:-dynamic main/0.
main:-
 catch((
   blint_version(Version),
   concat(['BLint v', Version], SVersion),
   current_prolog_flag(argv, Argv),
   parse_args(Argv, Filespec1),
   ifthen(not(show_in_console), 
    (writeln(SVersion),
     writeln(''))),   
   ifthenelse(is_sicstus, Filespec=Filespec1, Filespec1=[_|Filespec]), 
   !,
   main1(Filespec)
 ), E, (writeln(['Exception: ', E]), halt(9))),
 blint_exit.

main1([]):-
  show_usage,
  !.
main1(Filespec):-
  main(Filespec),
  !.

blint_exit:-
 call(any_errors_i),
 halt(1).
blint_exit:-
 halt(0).

show_usage:-
 on(Msg, [
 ['Usage:'],
 ['      BLint [Options] File [BLint_output_file]'],
 [''],
 ['generates {File}.blint unless BLint_output_file is provided'],
 [''],
 ['Options:'],
 ['        -ilntXXX Ignore lint error XXX (e.g. -ilnt102)'],
 ['        -dl Discard line information in lint output'],
 ['        -console Show lint output in console'],
 ['        -sl Sort lint output by line number'],
 ['        -backslash consider backslash as escape char'],
 ['        -dump Dump PrettyPrint file (extension: .PreetyPrint)'],
 ['        -ec Do not discard comments in pretty print file'],
 ['        -v Show version information'],
 ['        /X(XXX) Arity Lint ignore error XXX (e.g. /X(101))'],
 ['        /Q Same as -console'],
 ['']
 ]),
 concat(Msg, SMsg),
 writeln(SMsg),
 fail.
show_usage:- !.

blint_version('1.0').

parse_args([], []):- !.
parse_args([Arg|Argv], Filespec):-
  atom_chars(Arg, [Char|_]),
  on_test(Char, ['-', '/']),
  downcase_atom(Arg, LArg),
  parse_arg(LArg),
  !,
  parse_args(Argv, Filespec).
parse_args([Arg|Argv], [Arg|Filespec]):-
  !,
  parse_args(Argv, Filespec).

parse_arg('-sl'):-
  assert_sort_by_line_numbers,
  !.
parse_arg('-console'):-
  assert_show_in_console,
  !.
parse_arg('-backslash'):-
  assert_backslash_is_escape_char,
  !.
parse_arg('-ec'):-
  assert_enable_comments,
  !.
parse_arg('-dl'):-
  assert_discard_line_numbers,
  !.
parse_arg('-dump'):-
  assert_dump_file,
  !.
parse_arg('-v'):-
  blint_version(Version),
  concat(['BLint v', Version, ', (c)2009 Gustavo Brown - gbrown@fing.edu.uy'], SVersion),
  writeln(SVersion),
  writeln(''),
  !.
parse_arg(IgnoreLnt):-
  atom_chars(IgnoreLnt, ['-','i'|Chars]),
  atom_chars(LintId, Chars),
  assert_ignore_lint_id(LintId),
  !.
parse_arg(ArityArg):-
  %ALint compatibility
  atom_chars(ArityArg, ['/'|Chars]),
  parse_arg_arity(['/'|Chars]),
  !.
parse_arg(IgnoreArityLnt):-
  atom_chars(IgnoreArityLnt, ['/','x', '('|Chars]),  
  append(Chars1, [')'|Tail], Chars),
  atom_chars(ALintId, Chars1),
  parse_ignore_arity_lint_id(ALintId),
  ifthen(Tail\=[], (atom_chars(NIgnoreArityLnt, Tail), parse_arg(NIgnoreArityLnt))),
  !.
parse_arg(Arg):-
  concat(['** Invalid argument: ', Arg, '**'], SArg),
  writeln(SArg),
  writeln(''),
  !.

parse_arg_arity([]):- !.
parse_arg_arity([Char|Tail]):- 
  is_whitespace(Char),
  !,
  parse_arg_arity(Tail).
parse_arg_arity(['/','q'|Tail]):- 
  assert_show_in_console,
  !,
  parse_arg_arity(Tail).
parse_arg_arity(['/','x','('|Tail]):- 
  append(Chars1, [')'|Tail1], Tail),
  atom_chars(ALintId1, Chars1),
  atom_number(ALintId1, ALintId),  
  parse_ignore_arity_lint_id(ALintId),
  !,
  parse_arg_arity(Tail1).
parse_arg_arity(_):- !,fail.

parse_ignore_arity_lint_id(ALintId):-
  on([ALintId, LintId], [
      [101, lnt109],
      [101, lnt110],
      [103, lnt100],
      [106, lnt102],
      [203, lnt104],
      [204, lnt105],
      [208, lnt111],
      [210, lnt106],
      [303, lnt101],
      [308, lnt107],
      [309, lnt107]
    ]),
  assert_ignore_lint_id(LintId),
  fail.
parse_ignore_arity_lint_id(_):- !.


main([File, LintFile|_]):-
  blint_spec(File, LintFile),
  !.
main([File]):-
  blint_spec(File, []),
  !.
main([]):- !.

blint_spec(FileSpec, LintFile):-
  %expand_file_name(FileSpec, Files),
  absolute_file_name(FileSpec, File),
  blint(File, LintFile),
  fail.
blint_spec(_, _):- !.

blint(File1, LintFile):-
  initialize_blint,
  file_base_name_and_path(File1, Path, FileName),
  file_name_extension(FileBase, Ext, File1),
  ifthenelse(Ext='', (default_extension(DExt), concat([File1, '.', DExt], File)), File=File1),
  assert_default_extension(Ext),
  assert_default_path(Path),
  ifthen(not(show_in_console), 
    ( concat(['Processing file ''', FileName, ''''], PMsg),
      writeln(PMsg)
  )),

  open(File, read, In, [alias(input)]),
  ifthen(dump_file, (
    concat([FileBase, '.PrettyPrint'], OutFile),
    open(OutFile, write, Out, [alias(prettyprintfile)/*, close_on_abort(true)*/])
  )),
  ifthenelse(LintFile=[], concat([FileBase, '.blint'], BlintFile), BlintFile=LintFile),
  open(BlintFile, write, Blout, [alias(blintfile)/*, close_on_abort(true)*/]),

  blint_file_start(In),
  s_at_end_of_stream(In),
  blint_show_syntax_errors,
  ifthen(show_in_console, (current_output(Console), blint_file_out(Console, FileName))),
  blint_file_out(Blout, FileName),
  close(In),
  ifthen(dump_file, close(Out)),
  close(Blout),
  !.
blint(File, _):-
  writeln(['Error parsing file ''', File, '''''']),
  current_stream(_, read, Stream),
  close(Stream, [force(true)]),
  fail.
blint(_, _):- !.

blint_show_syntax_errors:-
  not(show_in_console),
  current_output(Out),
  on_lint_files(File),
  lint_error_id(_, LintId),
  any_lint(File, LintId, LintSId),
  concat(['< ', LintSId, ' >'], SLintId),
  writeln(Out, ''),
  writeln(Out, SLintId),
  call_lint(LintId, LineNro, File, Msg),
  blint_file_out2(Out, n, LineNro, File, Msg),
  fail.
blint_show_syntax_errors:- !.

initialize_blint:-
  s_clear_unget_buffer,
  initialize_line_count,
  initialize_pretty_printer,
  initialize_blint_facts,
  !.

:-dynamic enable_comments_i/0,
         ignore_lint_id_i/1,
         discard_line_numbers_i/0,
         sort_by_line_numbers_i/0,
         show_in_console_i/0,
         dump_file_i/0,
         backslash_is_escape_char_i/0,
         is_prolog_script_i/0,
         script_file_i/1,
         file_lints_i/1,
         default_extension_i/1,
         default_path_i/1,
         any_errors_i/0.

assert_any_errors:-
  assert(any_errors_i),
  !.

assert_ignore_lint_id(LintId):-
  call(lint_msg_text(LintId, _, _)),
  assert_once(ignore_lint_id_i(LintId)),
  !.
assert_ignore_lint_id(LntId):-
  concat(['Unknown lint id: ', LntId], Msg),
  debug_msg(Msg),
  !.

assert_sort_by_line_numbers:-
  assert_once(sort_by_line_numbers_i),
  !.

assert_show_in_console:-
  assert_once(show_in_console_i),
  !.

assert_enable_comments:-
  assert_once(enable_comments_i),
  debug_msg('[OPTN] Enable comments'),
  !.

assert_discard_line_numbers:-
  assert_once(discard_line_numbers_i),
%  debug_msg('[OPTN] Discard line numbers'),
  !.
  
assert_backslash_is_escape_char:-
  assert_once(backslash_is_escape_char_i),
  debug_msg('[OPTN] Backslash is escape char'),
  !.

assert_dump_file:-
  assert_once(dump_file_i),
  !.
  
assert_is_prolog_script:-
  assert_once(is_prolog_script_i),
  !.
  
assert_default_extension(Ext):-
  retractif(default_extension_i(_)),
  ifthen(Ext \= '', assert_once(default_extension_i(Ext))),
  !.

assert_default_path(Path):-
  retractif(default_path_i(_)),
  ifthen(Path \= '', assert_once(default_path_i(Path))),
  !.
  
default_extension(Ext):-
  call(default_extension_i(Ext)),
  !.
default_extension(ari):- !.

default_path(Path):-
  call(default_path_i(Path)),
  !.
default_path(''):- !.

ignore_lint_id(LintId):-
  call(ignore_lint_id_i(LintId)),
  !.

backslash_is_escape_char:-
  call(backslash_is_escape_char_i),
  !.
  
show_in_console:-
  call(show_in_console_i),
  !.
  
sort_by_line_numbers:-
  call(sort_by_line_numbers_i),
  !.

discard_comments:-
  call(enable_comments_i),
  !,
  fail.
discard_comments:- !.

dump_file:-
  call(dump_file_i),
  !.

discard_line_numbers:-
  call(discard_line_numbers_i),
  !.
  
is_prolog_script:-
  call(is_prolog_script_i),
  !.

blint_file_start(In):-
  peek_first_char(In, Char),
  Char = '#', % Veo si es un PrologScript
  get_until_eol(In, _), % Salto linea inicial del PrologScript
  assert_is_prolog_script,
  !,
  blint_file(In).
blint_file_start(In):-
  !,
  blint_file(In).


%blint_file(In):-
%% Hace un dump del archivo token-a-token
%  all_tokens(In, Token),
% !.
blint_file(In):-
  s_at_end_of_stream(In),
  blint_chk_endfile,
  !.
blint_file(In):-
  tokenize_one_pred(In, Pred),
  blint_file1(Pred),
  !,
  blint_file(In).

blint_file1([token(_, eof)]):- !.
blint_file1(Pred):-
%  debug_msg(['Predicado:']),
%  debug_msg_pred(Pred),
  pred_tokens(Pred, PredTokens, LineNo),
  parse_pred(ParsedPred1, PredTokens, []),
  parse_pred_errors(ParsedPred1, Pred, ParsedPred),
%  debug_msg(['ParsedPred:', ParsedPred]),
%  debug_msg(''),
  ifthen(dump_file, print_pred(ParsedPred)),
  parse_script(ParsedPred, NParsedPred),
  blint_pred(LineNo, NParsedPred),
  !.
blint_file1(Pred):-
% Errores parseando el predicado
  pred_tokens(Pred, PredTokens, LineNo),
  assert_lint(lnt203, LineNo, [PredTokens]),
  debug_msg(['Failed parsing predicate:']),
  debug_msg_pred(Pred),
  debug_msg(''),
  assert_any_errors,
  !.

parse_pred_errors([parseerror(Type, RLoc)], Pred, [parseerror(Type, ErrorLNo, SPred)]):-
  length(Pred, LPred),
  Loc is LPred-RLoc,
  parse_pred_errors1(Pred, Loc, 0, _, ErrorLNo, SPred1),
  concat(SPred1, SPred),
  !.
parse_pred_errors(ParsedPred, _, ParsedPred):- !.

parse_pred_errors1([], _, _, _, 0, []):- !.
parse_pred_errors1([token(ErrorLNo, Token)|Pred], Loc, Loc, CurLNo, ErrorLNo, [' << ', XToken, ' >> '|SPred]):-
  parse_pred_errors2(Token, ErrorLNo, CurLNo, XToken),
  !,
  parse_pred_errors1(Pred, 0, Loc, ErrorLNo, _, SPred).
parse_pred_errors1([token(TokenLNo, Token)|Pred], Loc, CurLoc, CurLNo, ErrorLNo, [XToken, ' '|SPred]):-
  succ(CurLoc, NCurLoc),
  parse_pred_errors2(Token, TokenLNo, CurLNo, XToken),
  !,
  parse_pred_errors1(Pred, Loc, NCurLoc, TokenLNo, ErrorLNo, SPred).

parse_pred_errors2(Token, CurLNo, CurLNo, XToken):-
  parse_pred_errors3(Token, XToken),
  !.
parse_pred_errors2(Token, _, _, XToken):-
  parse_pred_errors3(Token, XToken1),
  ifthenelse(XToken1 \= '', concat(['\n', XToken1], XToken), XToken=XToken1),
  !.

parse_pred_errors3(comment(_), ''):- !.
parse_pred_errors3(comment_line(_), ''):- !.
parse_pred_errors3(Token, XToken):-
  on_test([Token, XToken], [[comma, ','], [period, '.'], [cut, '!'], [openpar, '('], [closepar, ')'],
                            [startlist, '['], [endlist, ']'], [startsnip, '[!'], [endsnip,  '!]']]),
  !.
parse_pred_errors3(Token, XToken):-
  Token =.. [_,XToken],
  !.
parse_pred_errors3(Token, Token):- !.

parse_script([fact(prefix('?-', list(FilesList)))], []):-
  is_prolog_script,
  parse_script_files(FilesList),
  !.
parse_script([directive([list(FilesList)])], []):-
  parse_script_files(FilesList),
  !.
parse_script(ParsedPred, ParsedPred):- !.

parse_script_files([]):- !.
parse_script_files([Term|List]):-
  blint_term(atom, Term, [File|_]),
  compose_file(File, FileName),
  push_line_number(File),
  open(FileName, read, In, []),
  blint_file_start(In),
  pop_line_number,
  close(In),
  !,
  parse_script_files(List).

tokenize_one_pred(In, Pred):-
  in_c_directive,
  get_until_eol(In, Line),
  cur_line_ack(LNo),
  end_c_directive(LNo, Line, Pred),
  !.
tokenize_one_pred(In, Pred):-
  token_line(In, Token),
  !,
  tokenize_one_pred(Token, In, Pred).

end_c_directive(LNo, Line, [token(LNo, op((:-))), token(LNo, atom(prolog)), token(LNo, period)]):-
  trim_all_spaces(Line, Line1),
  atom_chars(':-prolog.', Line1),
  !.
end_c_directive(LNo, Line, [token(LNo, extrn(Line1))]):-
  atom_chars(Line1, Line),
  !.

trim_all_spaces([], []):- !.
trim_all_spaces([Char|List], SList):-
  is_whitespace(Char),
  !,
  trim_all_spaces(List, SList).
trim_all_spaces([Char|List], [Char|SList]):-
  !,
  trim_all_spaces(List, SList).

tokenize_one_pred(token(LNo, eof), _, [token(LNo, eof)]):- !.
tokenize_one_pred(token(LNo, period), _, [token(LNo, period)]):- !.
tokenize_one_pred(Token, In, [Token|Pred]):-
  token_line(In, NToken),
  !,
  tokenize_one_pred(NToken, In, Pred).

pred_tokens(Pred, PredTokens, LineNo):-
  findall(X, on(token(_, X), Pred), PredTokens),
  on_test(token(LineNo, _), Pred),
  !.
pred_tokens(_, [], 0):- !.

%all_tokens(In, Token):-
%  token_line(In, Token).
%all_tokens(In, Token):-
%  not(s_at_end_of_stream(In)),
%  !,
%  all_tokens(In, Token).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Lint tool                         %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

blint_file_out(Out, FileName):-
  bling_file_header(Out, FileName),
  ifthenelse(discard_line_numbers, DiscardLineNro=y, DiscardLineNro=n),
  blint_file_out1(Out, DiscardLineNro),
  !.

blint_file_out1(Out, _):-
  not(any_lint(_, _, _)),
  writeln(Out, 'No lint errors found'),
  writeln(Out, ''),
  ifthen(not(show_in_console), writeln('No lint errors found')),
  !.
blint_file_out1(Out, DiscardLineNro):-
  assert_any_errors,
  ifthen((not(show_in_console);(current_output(CurrentOutput), CurrentOutput=Out)), writeln('Lint errors found')),
  lint_ids(Ids),
  blint_file_body(Out, DiscardLineNro, Ids),
  !.

blint_file_out2(Out, n, LineNro, File, Msg):-
  LineNro \= 0,
  sfile_name(File, SFile),
  concat([SFile, 'Line ', LineNro, ': '], SMsg1),
  concat(['        ', Msg], SMsg2),
  writeln(Out, SMsg1),
  writeln(Out, SMsg2),
  !.
blint_file_out2(Out, _, _, File, Msg):-
  sfile_name(File, SFile),
  concat([SFile, '        ', Msg], SMsg),
  writeln(Out, SMsg),
  !.

sfile_name([], ''):- !.
sfile_name(File, SFile):-
  concat(['[', File, '] '], SFile),
  !.
  
blint_file_body(Out, DiscardLineNro, Ids):-
  sort_by_line_numbers,
  on_lint_files(File),
  findall(LineNro-LineNro, (on(LintId, Ids), call_lint(LintId, LineNro, File, _)), LLineNro),
  conj(LLineNro, LLineNro1),
  keysort(LLineNro1, LLineNro2),
  on(LineNro-_, LLineNro2),
  on(LintId, Ids),
  once(call_lint(LintId, LineNro, File, _)),
  blint_file_body_item(line, Out, LintId, DiscardLineNro, LineNro),
  fail.
blint_file_body(Out, DiscardLineNro, Ids):-
  not(sort_by_line_numbers),
  on(LintId, Ids),
  blint_file_body_item(ids, Out, LintId, DiscardLineNro, _),
  fail.
blint_file_body(_, _, _):- !.

blint_file_body_item(line, Out, LintId, DiscardLineNro, LineNro):-
  on_lint_files(File),
  any_lint(File, LintId, LintSId, LineNro),
  concat(['< ', LintSId, ' >'], SLintId),
  writeln(Out, ''),
  writeln(Out, SLintId),
  call_lint(LintId, LineNro, File, Msg),
  blint_file_out2(Out, DiscardLineNro, LineNro, File, Msg).
blint_file_body_item(ids, Out, LintId, DiscardLineNro, LineNro):-
  on_lint_files(File),
  any_lint(File, LintId, LintSId),
  concat(['< ', LintSId, ' >'], SLintId),
  writeln(Out, ''),
  writeln(Out, SLintId),
  call_lint(LintId, LineNro, File, Msg),
  blint_file_out2(Out, DiscardLineNro, LineNro, File, Msg).

bling_file_header(Out, Name):-
  writeln(Out, '---------------------------------------------------------------------------'),
  concat(['Blint: ', Name], SName),
  writeln(Out, SName),
  writeln(Out, '---------------------------------------------------------------------------'),
  writeln(Out, ''),
  !.

blint_pred(LineNo, ParsedPred):-
  call(cur_pred_i(CurPred)),
  blint_pred(LineNo, CurPred, ParsedPred),
  !.

blint_pred(_, _, [extrn(_)]):- !.
blint_pred(LineNo, _, [directive([Directive|Tail])]):-
  blint_directive(LineNo, [Directive|Tail]),
  !.
blint_pred(LineNo, CurPred, [fact(Fact)]):-
  blint_pred_goal(Fact, clause, Name/Arity),
  blint_pred1(LineNo, CurPred, Name/Arity, [Fact], clause),
  !.
blint_pred(LineNo, CurPred, [rule(Rule)|Body]):-
  blint_pred_goal(Rule, clause, Name/Arity),
  blint_pred1(LineNo, CurPred, Name/Arity, [Rule|Body], clause),
  !.
blint_pred(LineNo, CurPred, [dcg(Pred)|Body]):-
  blint_pred_goal(Pred, dcg, Name/Arity),
  blint_pred1(LineNo, CurPred, Name/Arity, [Pred|Body], dcg),
  !.
blint_pred(_, _, [parseerror(Type, LNo, SPred)]):-
  lint_error_id(Type, LntId),
  assert_lint(LntId, LNo, [SPred]),
  !.
blint_pred(LineNo, _, [_,_|_]):-
% Caso head con mas de un subgoal
  assert_lint(lnt105, LineNo, []),
  !.
blint_pred(_, _, []):-  !.

blint_pred1(LineNo, LastPred, CurPred, [Head|Body], PredType):-
  chk_contiguous_predicates(LineNo, LastPred, CurPred, ClauseNro),
  chk_predicate_redefinition(LineNo, CurPred),
  nb_setval(blineno, LineNo),
  blint_goal(Head, PredType, HeadVars, _),
  blint_goal(Body, PredType, BodyVars, BodyFunctors),
  conj(BodyFunctors, CalledPreds),
  blint_pred_used_predicates(CalledPreds, CurPred),
  blint_pred_vars(LineNo, CurPred, ClauseNro, HeadVars, BodyVars),
  !.

blint_pred_vars(LineNo, CurPred, ClauseNro, HeadVars, BodyVars):-
  append(HeadVars, BodyVars, AllVars),
  blint_get_singleton_vars(AllVars, SingletonVars),
  intersec(SingletonVars, HeadVars, HeadSingletonVars),
  intersec(SingletonVars, BodyVars, BodySingletonVars),
  blint_pred_vars1(LineNo, CurPred, ClauseNro, HeadSingletonVars, BodySingletonVars),
  !.

%blint_pred_vars1(LineNo, CurPred, ClauseNro, HeadSingletonVars, _):-
%  assert_lint(lnt109, LineNo, [debug, ClauseNro, CurPred]),
%  !.
blint_pred_vars1(LineNo, CurPred, ClauseNro, HeadSingletonVars, _):-
  HeadSingletonVars \= [],
  concat_c(HeadSingletonVars, ', ', SHeadSingletonVars1),
  concat(SHeadSingletonVars1, SHeadSingletonVars),
  assert_lint(lnt109, LineNo, [SHeadSingletonVars, ClauseNro, CurPred]),
  fail.
blint_pred_vars1(LineNo, CurPred, ClauseNro, _, BodySingletonVars):-
  BodySingletonVars \= [],
  concat_c(BodySingletonVars, ', ', SBodySingletonVars1),
  concat(SBodySingletonVars1, SBodySingletonVars),
  assert_lint(lnt110, LineNo, [SBodySingletonVars, ClauseNro, CurPred]),
  fail.
blint_pred_vars1(_, _, _, _, _):- !.

blint_get_singleton_vars([], []):- !.
blint_get_singleton_vars([Var|Vars], SingletonVars):-
  on_test(Var, Vars),
  findall(XVar, (on(XVar, Vars), XVar \= Var), Vars1),
  !,
  blint_get_singleton_vars(Vars1, SingletonVars).
blint_get_singleton_vars([Var|Vars], [Var|SingletonVars]):-
  !,
  blint_get_singleton_vars(Vars, SingletonVars).

blint_pred_used_predicates([], _):- !.
blint_pred_used_predicates([Pred|Predicates], CurPred):-
  assert(pred_used_i(Pred, CurPred)),
  !,
  blint_pred_used_predicates(Predicates, CurPred).

blint_pred_goal(Head, PredType, Name/Arity):-
  blint_goal(Head, PredType, _, [Name/Arity|_]),
  !.
blint_pred_goal(cut, clause, !/0):-!.
blint_pred_goal(cut, dcg, !/2):-!.

blint_directive(_, [atom(c)]):-
  assert_once(in_c_directive_i),
  !.
blint_directive(_, [atom(prolog)]):-
  ifthen(call(in_c_directive_i), retract(in_c_directive_i)),
  !.
blint_directive(LineNo, [prefix(Dir, DirHead)|Tail]):-
  blint_directive1(LineNo, Dir, [DirHead|Tail]),
  !.
blint_directive(_, [DirHead|_]):-
  blint_functor(DirHead, define, 1, _),
% Por ahora ignoramos la directive :-define  
  !.
blint_directive(_, [DirHead|_]):-
  blint_directive_op(DirHead),
  !.
blint_directive(_, [TDirHead|_]):-
  blint_term(infix, TDirHead, [_,DirHead,_]),
  blint_directive_op(DirHead),
  !.
blint_directive(LineNo, Directive):-
  blint_term_string(Directive, DirStr),
  assert_lint(lnt101, LineNo, [DirStr]),
  !.
  
blint_directive_op(DirHead):-
  blint_functor(DirHead, op, 3, [[TPrec], [TAssoc], [TOp]]),
  blint_term(number, TPrec, [Prec]),
  blint_term(atom, TAssoc, [Assoc]),
  blint_term(atom, TOp, [Op]),  
  add_op(Op, Assoc, Prec),
  !.  

:-dynamic in_c_directive_i/0.

in_c_directive:-
  call(in_c_directive_i),
  !.

blint_directive1(_, _, []):- !.
blint_directive1(LineNo, Dir, [DirHead|Tail]):-
  on_test(Dir, [public, visible, dynamic, extrn]),
  blint_pred_name(DirHead, Name, Arity),
  assert(dir_pred_access_i(Name/Arity, Dir)),
  !,
  blint_directive1(LineNo, Dir, Tail).
blint_directive1(_, module, _):- !.
blint_directive1(LineNo, mode, [_Dir|Tail]):-
%  blint_functor(Dir, Name, Arity, Params),
  !,
  blint_directive1(LineNo, mode, Tail).
blint_directive1(LineNo, Dir, [DirHead|Tail]):-
  blint_term_string(DirHead, Str),
  assert_lint(lnt102, LineNo, [Dir, Str]),
  !,
  blint_directive1(LineNo, Dir, Tail).

blint_chk_endfile:-
  chk_unused_predicates,
  chk_directives,
  !.

blint_goal(atom(Atom), clause, [], [Atom/0]):- !.
blint_goal(atom(Atom), dcg, [], [Atom/2]):- !.
blint_goal(var(Var), _, [Var], []):- !.
blint_goal(functor(Name, Arity, Params), clause, Vars, [Name/Arity|Functors]):-
  blint_goal_params(Params, Vars, Functors1),
  % Caso especial operador ifthen/2, ifthenelse/3 y alguno mas que ya se que se que hace call a los parametros
  ifthenelse(on_test(Name/Arity, [ifthen/2, ifthenelse/3, catch/3, not/1]), Functors=Functors1, Functors=[]),
  !.
blint_goal(functor(Name, Arity1, Params), PredType, Vars, [Name/Arity]):-
  blint_goal_params(Params, Vars, _),
  ifthenelse(PredType=dcg, Arity is Arity1+2, Arity=Arity1),
  !.
blint_goal(infix(Op, Left, Right), PredType, Vars, [Op/2|Functors]):-
  blint_goal(Left, PredType, LVars, LFunctors),
  blint_goal(Right, PredType, RVars, RFunctors),
  append(LVars, RVars, Vars),
  append(LFunctors, RFunctors, Functors),
  !.
blint_goal(prefix(Op, Right), PredType, Vars, [Op/1|Functors]):-
  blint_goal(Right, PredType, Vars, Functors),
  !.
blint_goal(postfix(Op, Left), PredType, Vars, [Op/1|Functors]):-
  blint_goal(Left, PredType, Vars, Functors),
  !.
blint_goal(list(List), _, Vars, []):-
  blint_goal(List, clause, Vars, _),
  !.
blint_goal(snip(Snip), PredType, Vars, Functors):-
  blint_goal(Snip, PredType, Vars, Functors),
  !.
blint_goal([Term|Tail], PredType, Vars, Functors):-
  blint_goal_paren([Term|Tail], PredType, Vars, Functors),
  !.
blint_goal([], _, [], []):- !.
blint_goal(dcgprolog(DCG), _, Vars, Functors):-
  blint_goal(DCG, clause, Vars, Functors),
  !.
blint_goal(extrn(XBody), clause, Vars, _):-
  parse_extrn(XBody, Parsed),
  blint_goal(Parsed, clause, Vars, _),
  nb_getval(blineno, LineNo),
  assert_lint(lnt111, LineNo, []),
  !.
blint_goal(Term, _, [], []):-
  Term =.. [TermType|_],
  on_test(TermType, [unnamed_var, number, cut, extrn]),
  !.
blint_goal(comment(_, Goal), PredType, Vars, Functors):-
  !,
  blint_goal(Goal, PredType, Vars, Functors).
blint_goal(comment_line(_, Goal), PredType, Vars, Functors):-
  !,
  blint_goal(Goal, PredType, Vars, Functors).
blint_goal(X, _, [], []):-
  debug_msg(['Failed blint_goal: ', X]),
  !.

blint_goal_params([], [], []):- !.
blint_goal_params([param(Param)|Params], Vars, Functors):-
  blint_goal(Param, clause, PVars, PFunctors),
  blint_goal_params(Params, BVars, BFunctors),
  append(PVars, BVars, Vars),
  append(PFunctors, BFunctors, Functors),
  !.

blint_goal_paren([], _, [], []):- !.
blint_goal_paren([Goal|Tail], PredType, Vars, Functors):-
  blint_goal(Goal, PredType, GVars, GFunctors),
  blint_goal_paren(Tail, PredType, TVars, TFunctors),
  append(GVars, TVars, Vars),
  append(GFunctors, TFunctors, Functors),
  !.

blint_functor(TFunctor, Name, Arity, Params):-
  blint_term(functor, TFunctor, [Name, Arity, TParams]),
  blint_params(TParams, Params),
  !.

blint_params([], []):- !.
blint_params([TParam|TParams], [Param|Params]):-
  blint_term(param, TParam, Param),
  !,
  blint_params(TParams, Params).

blint_pred_name(Pred, Name, Arity):-
  blint_pred_name1(Pred, Pred1),
  blint_term(infix, Pred1, [/, TName, TArity]),
  blint_term(atom, TName, [Name]),
  blint_term(number, TArity, [Arity]),
  !.
  
blint_pred_name1(Pred, Pred2):-  
  blint_term(infix, Pred, [':', Pred1|_]),
  blint_pred_name1(Pred1, Pred2),
  !.
blint_pred_name1(Pred, Pred):- !.  

blint_term_string(atom(Atom), Atom):- !.
blint_term_string(var(Var), Var):- !.
blint_term_string(unnamed_var(Var), Var):- !.
blint_term_string(number(Number), Number):- !.
blint_term_string(cut, '!'):- !.
blint_term_string(functor(Name, _, Params), Functor):-
  blint_term_params_string(Params, '', SParams),
  concat([Name, '(', SParams, ')'], Functor),
  !.
blint_term_string(infix(Op, Left, Right), Infix):-
  blint_term_string(Left, SLeft),
  blint_term_string(Right, SRight),
  ifthenelse(atom_chars(Op, ['/']),
    concat([SLeft, '/', SRight], Infix),
    concat([SLeft, ' ', Op, ' ', SRight], Infix)
  ),
  !.
blint_term_string(prefix(Op, Right), Prefix):-
  blint_term_string(Right, SRight),
  atom_chars(Op, [Char|_]),
  ifthenelse(is_alpha(Char),
    concat([Op, ' ', SRight], Prefix),
    concat([Op, SRight], Prefix)
  ),
  !.
blint_term_string(postfix(Op, Left), Postfix):-
  blint_term_string(Left, '', SLeft),
  atom_chars(Op, OpChars),
  reverse(OpChars, [Char|_]),
  ifthenelse(is_alpha(Char),
    concat([' ', Op, SLeft], Postfix),
    concat([Op, SLeft], Postfix)
  ),
  !.
blint_term_string(list(List), SList):-
  blint_term_list_items_string(List, '', '', SListItems),
  concat(['[', SListItems, ']'], SList),
  !.
blint_term_string(snip(Snip), SSnip):-
  blint_term_paren_string(Snip, '', SListItems),
  concat(['[! ', SListItems, '!]'], SSnip),
  !.
blint_term_string([Term|Tail], SParen):-
  blint_term_paren_string([Term|Tail], '', SListItems),
  concat(['( ', SListItems, ')'], SParen),
  !.
blint_term_string([], ''):- !.
blint_term_string(extrn(Code), Code):- !.
blint_term_string(dcgprolog([DCG]), SDCG):-
  blint_term_paren_string(DCG, '', SListItems),
  concat(['{ ', SListItems, '}'], SDCG),
  !.
blint_term_string(comment(Comment, Term), STerm):-
  blint_term_string(Term, STerm1),
  concat(['/* ', Comment, ' */', STerm1], STerm),
  !.
blint_term_string(comment_line(Comment, Term), STerm):-
  blint_term_string(Term, STerm1),
  concat(['/* ', Comment, ' */', STerm1], STerm),
  !.
blint_term_string(X, X):-
  debug_msg(''),
  debug_msg(['Failed blint_term_string: ', X]),
  !.

blint_term_params_string([], SParams, SParams):- !.
blint_term_params_string([param(Term)|Params], SParams1, SParams):-
  blint_term_string(Term, STerm),
  ifthenelse(Params\=[],
    concat([SParams1, STerm, ', '], SParams2),
    concat([SParams1, STerm], SParams2)
  ),
  !,
  blint_term_params_string(Params, SParams2, SParams).
  
blint_term_paren_string([], SParams, SParams):- !.
blint_term_paren_string([Term|Params], SParams1, SParams):-
  blint_term_string(Term, STerm),
  ifthenelse(Params\=[],
    concat([SParams1, STerm, ', '], SParams2),
    concat([SParams1, STerm], SParams2)
  ),
  !,
  blint_term_paren_string(Params, SParams2, SParams).  

blint_term_list_items_string([], _, SListItems, SListItems):- !.
blint_term_list_items_string([list([Tail])], _, SListItems1, SListItems):-
  blint_term_string(Tail, STerm),
  concat([SListItems1, '|', STerm], SListItems),
  !.
blint_term_list_items_string([Term|Tail], Sep, SListItems1, SListItems):-
  blint_term_string(Term, STerm),
  concat([SListItems1, Sep, STerm], SListItems2),
  !,
  blint_term_list_items_string(Tail, ', ', SListItems2, SListItems).

blint_term(number, number(Num1), [Num]):-
  atom_number(Num1, Num),
  !.
blint_term(TermType, Term, Args):-
  Term =..[TermType|Args],
  !.

chk_predicate_redefinition(LineNo, Op/Arity):-
  operator_type(Op, OpType, _),
  on_test([OpType, Arity], [
     [xfx, 2],
     [xfy, 2],
     [yfx, 2],
     [xf, 1],
     [yf, 1],
     [fx, 1],
     [fy, 1]
  ]),
  assert_lint(lnt103, LineNo, [Op/Arity]),
  !.
chk_predicate_redefinition(LineNo, CurPred):-
  builtin_predicate(CurPred, _),
  assert_lint(lnt104, LineNo, [CurPred]),
  !.
chk_predicate_redefinition(LineNo, !/0):-
  assert_lint(lnt103, LineNo, [(!)/0]),
  !.
chk_predicate_redefinition(_, _):- !.

chk_contiguous_predicates(_, Pred, Pred, ClauseNro):-
  set_cur_pred(Pred, ClauseNro),
  !.
chk_contiguous_predicates(_, none, Pred, ClauseNro):-
  set_cur_pred(Pred, ClauseNro),
  !.
chk_contiguous_predicates(LineNo, _, Pred, ClauseNro):-
  call(parsed_pred_i(Pred, _)),
  set_cur_pred(Pred, ClauseNro),
  assert_lint(lnt100, LineNo, [Pred]),
  !.
chk_contiguous_predicates(_, _, Pred, ClauseNro):-
  set_cur_pred(Pred, ClauseNro),
  !.

chk_directives:-
  call(dir_pred_access_i(Pred, Dir)),
  chk_directives1(Pred, Dir),
  fail.
chk_directives:-
  chk_directive_archive_inaccesible,
  !.

chk_directive_archive_inaccesible:-
  call(dir_pred_access_i(_, public)),
  !.
chk_directive_archive_inaccesible:-
  call(dir_pred_access_i(_, visible)),
  !.
chk_directive_archive_inaccesible:-
  call(dir_pred_access_i(_, dynamic)),
  !.
chk_directive_archive_inaccesible:-
  assert_lint(lnt108, 0, []),
  !.

chk_directives1(_, Dir):- 
  on_test(Dir, [extrn, dynamic]), 
  !.
chk_directives1(_, dynamic):- !.
chk_directives1(Pred, _):-
  call(parsed_pred_i(Pred, _)),
  !.
chk_directives1(Pred, Dir):-
  assert_lint(lnt107, 0, [Dir, Pred]),
  !.

chk_unused_predicates:-
  call(parsed_pred_i(Pred, _)),
  chk_unused_predicates1(Pred),
  fail.
chk_unused_predicates:- !.

chk_unused_predicates1(Pred):-
  call(dir_pred_access_i(Pred, Dir)),
  on_test(Dir, [public, visible, dynamic]),
  !.
chk_unused_predicates1(Pred):-
  call(pred_used_i(Pred, FromPred)),
  Pred \= FromPred,
  !.
chk_unused_predicates1(Pred):-
  assert_lint(lnt106, 0, [Pred]),
  !.

builtin_predicate(Functor/Arity, Mode):-
  on([Functor/Arity, Mode], [
    [abolish/1, [+]], [abort/1, [+]], [api_mono/0, []], [arg/3, [+,+,-]], [arg0/3, [+,+,-]],
    [argrep/4, [+,+,+,-]], [assert/1, [+]], [assert/2, [+,+]], [asserta/1, [+]], [assertz/1, [+]],
    [atom/1, [?]], [atom_string/2, [?,?]], [atomic/1, [?]], [bagof/3, [?,+,-]], [betweenb/7, [+,?,?,?,?,?,?]],
    [betweenkeysb/4, [+,?,?,?]], [bit_count/2, [+,-]], [bit_lit/2, [+,+]], [bit_unlit/2, [+,+]],
    [break/0, []], [btree_count/2, [+,-/+]], [bwscreen/0, []], [call/1, [+]], [call/2, [+,+]],
    [case/1, [+]], [chdir/1, [?]], [chmod/2, [+,?]], [clause/2, [+,-]], [close/1, [+]], [cls/0, []],
    [code_world/2, [?,?]], [command_string/1, [-]], [compare/3, [-,+,+]], [compare/4, [-,+,+,+]],
    [concat/3, [+,+,-]], [concat/2, [+,-]], [consult/1, [+]], [create/2, [-,+]], [create_popup/4, [?/+,?/+,?/+,?/+]],
    [create_popup/5, [?/+,?,?,?,?/+]], [create_world/1, [+]], [ctr_dec/2, [+,-]], [ctr_inc/2, [+,-]],
    [ctr_is/2, [+,-]], [ctr_set/2, [+,+]], [current_op/3, [?,?,?]], [current_predicate/1, [?]],
    [current_window/2, [-,?]], [data_world/2, [?,?]], [date/1, [?]], [date_day/2, [+,-]], [dbcs2term/1, [+,-]],
    [dbPartition/2, [-,+]], [dbs2term/2, [+,-]], [dbsLen/2, [+,-]], [debug/0, []], [debug/1, [+]], [debugger/0, []],
    [dec2, [+,-]], [define_window/5, [+,?/+,?,?,?]], [define_window/6, [+,?/+,?,?,?,?/+]], [defineb/4, [+,+,+,+]],
    [defineb/5, [+,+,+,+,+]], [defineh/2, [+,+]], [delete/1, [+]], [delete_window/1, [+]], [delete_world/1, [+]],
    [dialog_run/1, [+]], [dialog_run/2, [+,+]], [dialog_run/3, [+,+,+]], [directory/6, [+,-,-,-,-,-]],
    [disk/1, [?]], [display/1, [+]], [display/2, [+,+]], [dll_load/3, [+,-,-]], [dll_free/1, [+]], [dll_handle/2, [?,?]],
    [dll_address/3, [+,+,-]], [dll_visi/2, [+,+]], [edit/1, [+]], [eq/2, [?,?]], [erase/1, [+]], [eraseall/1, [+]],
    [errcode/2, [-,+]], [evalAddress/2, [+,+]], [exit_dbox/1, [+]], [exit_popup/0, []], [expand_term/2, [+,-]],
    [expunge/0, []], [fail/0, []], [file_list/1, [+]], [file_list/2, [+,+]], [fileerrors/2, [?,?]],
    [findall/3, [?,+,-]], [float/1, [?]], [float_text/3, [?,?,?]], [flush/0, []], [functor/3, [?,?,?]], [gc/0, []],
    [get/1, [?]], [get/2, [+,?]], [get0/0, [?]], [get0/2, [+,?]], [get0_noecho/1, [?]], [get_bit/3, [+,+,-]],
    [get_cursor/2, [-,-]], [getEnvVar/2, [+,-/+]], [ground/1, [?]], [halt/0, []], [halt/1, [+]], [hard_erase/1, [+]],
    [hide_window/2, [?,+]], [ifthen/2, [+,+]], [ifthenelse/3, [+,+,+]], [in/2, [+,?]], [inc/2, [+,-]],
    [instance/2, [+,-]], [int_text/2, [?,?]], [integer/1, [?]], [key/2, [+,-]], [key_count/2, [+,-]], [keyb/2, [-,-]],
    [keyb/3, [-,-,-/+]], [keyb_peek/2, [-,-]], [keyb_peek/3, [-,-,-/+]], [keys/1, [?]], [keysort/2, [+,-]],
    [leash/1, [+]], [length/2, [+,-]], [list_text/2, [?,?]], [listing/0, []], [listing/1, [+]], [load_key/2, [+,+]],
    [lock/0, []], [mkdir/1, [+]], [move_window/2, [+,+]], [mth_ref/3, [+,+,-]], [name/2, [?,?]], [nl/0, []],
    [nl/1, [+]], [nodebug/0, []], [nodebug/1, [-]], [nonvar/1, [?]], [nospy/1, [+]], [not/1, [+]], [notrace/0, []],
    [notrace/1, [+]], [nref/2, [+,-]], [nth_char/3, [+,-,-]], [nth_ref/3, [+,+,-]], [number/1, [?]], [op/3, [+,+,+]],
    [open/3, [-,+,+]], [out/2, [+,+]], [p_open/3, [-/+,+,+]], [p_open/4, [-,+,+,+]], [pfname/5, [+,?/+,?/+,?/+,?/+]],
    [pref/2, [+,-]], [put/1, [+]], [put/2, [+,+]], [randomize/1, [+]], [read/1, [-]], [read/2, [+,-]],
    [read_ascib/2, [+,+]], [read_asciw/2, [+,+]], [read_asciz/2, [+,+]], [read_asciz_field/3, [+,+,+]],
    [read_float/2, [+,+]], [read_int8/2, [+,+]], [read_int16/2, [+,+]], [read_int32/2, [+,+]], [read_line/2, [+,-]],
    [read_string/2, [+,-]], [read_string/3, [+,+,-]], [recolor_window/2, [+,+]], [reconsult/1, [+]],
    [record_after/3, [+,?,-]], [record_before/3, [+,?,-]], [record_visi/3, [+,+,+]], [recorda/3, [+,?,-]],
    [recordb/3, [+,+,+]], [recorded/3, [+,?,-]], [recorded/4, [+,+,?,-]], [recorded_nth/4, [+,+,-,-]],
    [recorded_ref/4, [+,+,?,-]], [recorded_terms/3, [+,?,-]], [recorded_tro/3, [+,?,-]], [recordh/3, [+,+,+]],
    [recordz/3, [+,?,-]], [ref/1, [?]], [refPartition/2, [+,-]], [refresh/0, []], [region_c/3, [+,+,?]],
    [region_ca/3, [+, +, ?]], [region_cc/3, [+,+,?]], [relabel_window/1, [+]], [removeallb/1, [+]],
    [removeallh/1, [+]], [removeb/3, [+,+,?]], [removeb/2, [+,+]], [removeh/3, [+,+,?]], [rename/2, [+,+]],
    [repeat/0, []], [replace/2, [+,+]], [replaceb/4, [+,+,+,+]], [replaceh/4, [+,+,?,+]], [repos_window/2, [+,+]],
    [reset_op/0, []], [resetspy/0, []], [resize_window/2, [+,+]], [restore/0, []], [restore/1, [+]],
    [retract/1, [+]], [retrieveb/3, [+,?,?]], [retrieveb_keys/2, [+,-]], [retrieveb_nth/3, [+,+,-]],
    [retrieveb_terms/2, [+,-]], [retrieveb_terms/3, [+,+,-]], [retrieveh/3, [+,?,?]],
    [rmdir/1, [+]], [save/0, []], [save/1, [+]], [scan_code/2, [+,-]], [screen_height/2, [?/+,?/+]],
    [searchPath/4, [+,+,+,-/+]], [see/1, [+]], [see_h/1, [+]], [seeing/1, [?]], [seek/4, [+,+,+,-]], [seen/0, []],
    [send_control_msg/3, [+,+,+]], [send_dialog_msg/3, [+,+,+]], [send_menu_msg/2, [+,-]], [set_bit/5, [+,+,+,-,-]],
    [set_bits/4, [+,+,+,-]], [set_control/1, [+]], [set_cursor/2, [+,+]], [set_cursorshape/1, [+]], [setof/3, [?,+,-]],
    [setup_editor/6, [+,+,+,+,+,+]], [shell/0, []], [shell/1, [+]], [skip/1, [+]], [skip/2, [+,+]], [sort/2, [+,-]],
    [sortkey/1, [+]], [spy/1, [+]], [statistics/0, []], [statistics/2, [+,-]], [stdin/2, [+,+]], [stdinout/3, [+,+,+]],
    [stdout/2, [+,+]], [store_windows/0, []], [string/1, [?]], [string_length/2, [+,?]], [string_lower/2, [+,-]],
    [string_search/3, [+,+,-]], [string_search/4, [+,+,+,-]], [string_term/2, [?,?]], [string_termq/2, [?/+,?/+]],
    [string_upper/2, [+,-]], [substring/4, [+,+,+,-]], [syntaxerrors/2, [?,?]], [system/1, [?]], [system/3, [?,-,+]],
    [sysflag/3, [+,-,+]], [tab/1, [+]], [tab/2, [+,+]], [tchar/2, [-,-]], [tell/1, [+]], [tell_h/1, [+]], [telling/1, [?]],
    [term2dbcs/4, [+,+,+,-]], [term2dbs/2, [+,-]], [term_concat/3, [+,+,-]], [tget/2, [?,?]], [tget_screen/2, [?,?]],
    [time/1, [?]], [tmove/2, [+,+]], [tmove_screen/2, [+,+]], [told/0, []], [trace/0, []], [trace/1, [+]], [true/0, []],
    [tscroll/3, [+,+,+]], [unlock/0, []], [var/1, [?]], [varnames/1, [?]], [wa/2, [+,+]], [wc/2, [+,+]], [wca/3, [+,+,+]],
    [what_btrees/1, [-]], [what_windows/1, [?]], [what_worlds/1, [?]], [which_control/1, [-]], [window_info/5, [?/+,?/+,?/+,?/+,?/+]],
    [window_info/6, [?/+,?/+,?/+,?/+,?/+,?/+]], [workspace/2, [-,+]], [write/1, [+]], [write/2, [+,+]], [write_ascib/2, [+,+]],
    [write_asciw/2, [+,+]], [write_asciz/2, [+,+]], [write_asciz_field/3, [+,+,+]], [write_ctrl_text/2, [+,+]],
    [write_float/2, [+,+]], [write_int8/2, [+,+]], [write_int16/2, [+,+]], [write_int32/2, [+,+]], [write_key/3, [+,+,+]],
    [writeq/1, [+]], [writeq/2, [+,+]]
  ]).

set_cur_pred(CurPred, ClauseNro):-
  retract(cur_pred_i(_)),
  assert(cur_pred_i(CurPred)),
  retract(parsed_pred_i(CurPred, ClauseNro1)),
  succ(ClauseNro1, ClauseNro),
  assert(parsed_pred_i(CurPred, ClauseNro)),
  !.
set_cur_pred(CurPred, 1):-
  assert(parsed_pred_i(CurPred, 1)),
  !.

assert_lint(LintId, _, _):-
  ignore_lint_id(LintId),
  !.
assert_lint(LintId, LineNo, Parms):-
  lint_msg(LintId, Parms, Msg),
  get_current_file(File),
  assert_once(file_lints_i(File)),
  assert_once(lint_i(LintId, LineNo, File, Msg)),
  !.

call_lint(LintId, LineNo, File, Msg):-
  call(lint_i(LintId, LineNo, File, Msg)).

lint_msg(LintId, Parms, Msg):-
  lint_msg_text(LintId, _, Text),
  replace_args(Text, Parms, Msg1),
  concat(Msg1, Msg),
  !.

replace_args([], _, []):- !.
replace_args([ArgNo|Tail], Parms, [Arg|Msg]):-
  integer(ArgNo),
  nth_arg(ArgNo, 1, Parms, Arg1),
  replace_args1(Arg1, Arg),
  !,
  replace_args(Tail, Parms, Msg).
replace_args([Arg|Tail], Parms, [Arg|Msg]):-
  !,
  replace_args(Tail, Parms, Msg).

replace_args1(Arg, Arg):-
  atomic(Arg),
  !.
replace_args1(Arg1, Arg):-
  term_to_atom(Arg1, Arg),
  !.

nth_arg(_, _, [], '{missing arg}'):- !.
nth_arg(ArgNo, ArgNo, [Arg|_], Arg):-!.
nth_arg(ArgNo, CurArg, [_|Parms], Arg):-
  succ(CurArg, NArg),
  !,
  nth_arg(ArgNo, NArg, Parms, Arg).

any_lint(File, LintId, LintSId):-
  any_lint(File, LintId, LintSId, _),
  !.
  
any_lint(File, LintId, LintSId, LineNro):-
  call_lint(LintId, LineNro, File, _),
  ifthen(LineNro==0, File==[]),
  lint_msg_text(LintId, LintSId, _),
  !.
  
on_lint_files(File):-
        findall(F, file_lints_i(F), Files1),
        conj(Files1, Files),
        on(File, Files).

lint_error_id(Type, LntId):-
  on([Type, LntId], [
    [parse, lnt200],
    [syntax, lnt201],
    [list_syntax, lnt202]
  ]).

lint_ids(LintIds):-
  findall(LintId, call(lint_msg_text(LintId, _, _)), LintIds),
  !.   

% Mensajes del lint
lint_msg_text(lnt100, 'LNT 100', ['Predicate <<', 1, '>> not contiguous.']).
lint_msg_text(lnt101, 'LNT 101', ['Unknown directive <<', 1, '>>.']).
lint_msg_text(lnt102, 'LNT 102', ['Cannot parse directive <<', 1, '>> argument <<', 2, '>>.']).
lint_msg_text(lnt103, 'LNT 103', ['Operator <<', 1, '>> redefined.']).
lint_msg_text(lnt104, 'LNT 104', ['Builtin predicate  <<', 1, '>> redefined.']).
lint_msg_text(lnt105, 'LNT 105', ['Cannot redefine operator (,)/2.']).
lint_msg_text(lnt106, 'LNT 106', ['Predicate <<', 1, '>> is not used anywhere.']).
lint_msg_text(lnt107, 'LNT 107', ['Directive <<', 1, '>> declared for predicate <<', 2, '>> but no definition found.']).
lint_msg_text(lnt108, 'LNT 108', ['No public/visible predicates found.']).
lint_msg_text(lnt109, 'LNT 109', ['Singleton variables <<', 1, '>> found in the head of clause ', 2, ' of <<', 3, '>>.']).
lint_msg_text(lnt110, 'LNT 110', ['Singleton variables <<', 1, '>> found in the body of clause ', 2, ' of <<', 3, '>>.']).
lint_msg_text(lnt111, 'LNT 111', ['Extern code expressions not analyzed.']).
lint_msg_text(lnt200, 'LNT 200', ['Could not parse predicate:\n', 1, '\n']).
lint_msg_text(lnt201, 'LNT 201', ['Syntax error parsing predicate:\n', 1, '\n']).
lint_msg_text(lnt202, 'LNT 202', ['Syntax error in list:\n', 1, '\n']).
lint_msg_text(lnt203, 'LNT 203', ['Could not parse predicate <<', 1, '>>.']).

:-dynamic lint_i/4,
          cur_pred_i/1,
          parsed_pred_i/2,
          dir_pred_access_i/2,
          pred_used_i/2.

initialize_blint_facts:-
  retractall(cur_pred_i(_)),
  retractall(parsed_pred_i(_,_)),
  retractall(dir_pred_access_i(_,_)),
  retractall(pred_used_i(_,_)),
  retractall(lint_i(_,_,_,_)),
  retractall(file_lints_i(_)),
  retractall(in_c_directive_i),
  retractall(op_type_i(_,_,_)),

  assert(cur_pred_i(none)),
  !.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Pretty Printer                    %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

print_pred(ParsedPred):-
  stream_property(Out, alias(prettyprintfile)),
  print_pred(Out, ParsedPred),
  !.

print_pred(Out, [directive([Directive|Tail])]):-
  print_cur_pred(Out, directive),
  write(Out, ':-'),
  print_term(Out, Directive, 0),
  print_term_list(Out, Tail, 9),
  writeln(Out, '.'),
  !.
print_pred(Out, [fact(Fact)]):-
  print_cur_pred(Out, Fact),
  print_term(Out, Fact, 0),
  writeln(Out, '.'),
  !.
print_pred(Out, [rule(Rule)|Body]):-
  print_cur_pred(Out, Rule),
  print_term(Out, Rule, 0),
  writeln(Out, ':-'),
  print_pred1(Out, Body, 2),
  writeln(Out, '.'),
  !.
print_pred(Out, [dcg(Pred)|Body]):-
  print_cur_pred(Out, Pred),
  print_term(Out, Pred, 0),
  write(Out, '-->'),
  ifthen(Body=[_,_|_], writeln(Out, '')),
  print_pred1(Out, Body, 2),
  writeln(Out, '.'),
  !.
print_pred(Out, [X,Y|Z]):-
% Caso Head con mas de 1 subgoal
  print_pred1(Out, [X,Y|Z], 0),
  writeln(Out, '.'),
  !.
print_pred(Out, [extrn(Code)]):-
  writeln(Out, Code),
  !.
print_pred(Out, [parseerror(_, _, Code)]):-
  writeln(Out, Code),
  !.
print_pred(_, []):-  !.
%print_pred(_, _):-  !.

print_pred1(_, [], _):- !.
print_pred1(Out, [Term|Body], Spaces):-
  print_spaces(Out, Spaces),
  print_term(Out, Term, Spaces),
  ifthen(Body\=[], writeln(Out, ',')),
  !,
  print_pred1(Out, Body, Spaces).

print_term(Out, atom(Atom), _):-  write(Out, Atom), !.
print_term(Out, var(Var), _):- write(Out, Var), !.
print_term(Out, unnamed_var(Var), _):- write(Out, Var), !.
print_term(Out, number(Number), _):- write(Out, Number), !.
print_term(Out, cut, _):- write(Out, '!'), !.
print_term(Out, functor(Name, _, Params), Spaces):-
  write(Out, Name),
  write(Out, '('),
  print_params(Out, Params, Spaces),
  write(Out, ')'),
  !.
print_term(Out, infix(Op, Left, Right), Spaces):-
  print_term(Out, Left, Spaces),
  ifthenelse(atom_chars(Op, ['/']),
    write(Out, '/'),
    ( write(Out, ' '),
      write(Out, Op),
      write(Out, ' ')
  )),
  print_term(Out, Right, Spaces),
  !.
print_term(Out, prefix(Op, Right), Spaces):-
  write(Out, Op),
  atom_chars(Op, [Char|_]),
  ifthen(is_alpha(Char), write(Out, ' ')),
  print_term(Out, Right, Spaces),
  !.
print_term(Out, postfix(Op, Left), Spaces):-
  print_term(Out, Left, Spaces),
  atom_chars(Op, OpChars),
  reverse(OpChars, [Char|_]),
  ifthen(is_alpha(Char), write(Out, ' ')),
  write(Out, Op),
  !.
print_term(Out, list(List), Spaces):-
  write(Out, '['),
  print_list_items(Out, List, Spaces, ''),
  write(Out, ']'),
  !.
print_term(Out, snip([Term|Tail]), Spaces):-
  write(Out, '[! '),
  Spaces1 is Spaces+2,
  print_term_paren(Out, Term, Tail, Spaces, Spaces1),
  ifthen(Tail=[], write(Out, ' ')),
  write(Out, '!]'),
  !.
print_term(Out, [Term|Tail], Spaces):-
  write(Out, '('),
  Spaces1 is Spaces+2,
  print_term_paren(Out, Term, Tail, Spaces, Spaces1),
  write(Out, ')'),
  !.
print_term(Out, extrn(Code), _):-
  write(Out, Code),
  !.
print_term(Out, dcgprolog([Term|Tail]), Spaces):-
  write(Out, '{'),
  Spaces1 is Spaces+2,
  print_term_paren(Out, Term, Tail, Spaces, Spaces1),
  write(Out, '}'),
  !.
print_term(Out, comment(Comment, Term), Spaces):-
  write(Out, '/*'),
  write(Out, Comment),
  writeln(Out, '*/'),
  print_spaces(Out, Spaces),
  print_term(Out, Term, Spaces),
  !.
print_term(Out, comment_line(Comment, Term), Spaces):-
  write(Out, '%'),
  writeln(Out, Comment),
  print_spaces(Out, Spaces),
  print_term(Out, Term, Spaces),
  !.
print_term(_, X, _):-
  debug_msg(''),
  debug_msg(['Failed printing term: ', X]),
  !,
  fail.

print_term_paren(Out, Term, [], _, Spaces1):-
  print_term(Out, Term, Spaces1),
  !.
print_term_paren(Out, Term, Tail, Spaces, Spaces1):-
  writeln(Out, ''),
  print_spaces(Out, Spaces1),
  print_term(Out, Term, Spaces1),
  print_term_list(Out, Tail, Spaces1),
  writeln(Out, ''),
  print_spaces(Out, Spaces),
  !.

print_term_list(_, [], _):- !.
print_term_list(Out, [Term|Body], Spaces):-
  writeln(Out, ','),
  print_spaces(Out, Spaces),
  print_term(Out, Term, Spaces),
%  ifthen(Body\=[], writeln(Out, ',')),
  !,
  print_term_list(Out, Body, Spaces).

print_params(_, [], _):- !.
print_params(Out, [param(Term)|Params], Spaces):-
  print_term(Out, Term, Spaces),
  ifthen(Params\=[], write(Out, ', ')),
  !,
  print_params(Out, Params, Spaces).

print_list_items(_, [], _, _):- !.
print_list_items(Out, [Term|Tail], Spaces, Sep):-
  write(Out, Sep),
  print_term(Out, Term, Spaces),
  !,
  print_list_items(Out, Tail, Spaces, ', ').

print_spaces(_, 0):- !.
print_spaces(Out, Spaces):-
  Spaces1 is Spaces-1,
  write(Out, ' '),
  !,
  print_spaces(Out, Spaces1).

:-dynamic pp_cur_pred_i/1.

initialize_pretty_printer:-
  retractall(pp_cur_pred_i(_)),
  assert(pp_cur_pred_i(none)),
  !.

print_cur_pred(Out, directive):-
  retract(pp_cur_pred_i(CurPred)),
  print_cur_pred_directive(Out, CurPred),
  assert(pp_cur_pred_i(directive)),
  !.
print_cur_pred(Out, Pred):-
  print_cur_pred1(Pred, Name, Arity),
  call(pp_cur_pred_i(CurPred)),
  print_cur_pred2(Out, Name, Arity, CurPred),
  !.
print_cur_pred(_, _):- !.

print_cur_pred1(atom(Name), Name, 0):- !.
print_cur_pred1(functor(Name, Arity,_), Name, Arity):- !.

print_cur_pred2(_, Name, Arity, Name/Arity):- !.
print_cur_pred2(Out, Name, Arity, CurPred):-
  retract(pp_cur_pred_i(CurPred)),
  assert(pp_cur_pred_i(Name/Arity)),
  ifthen(CurPred \= none, writeln(Out, '')),
  !.

print_cur_pred_directive(_, directive):- !.
print_cur_pred_directive(_, none):- !.
print_cur_pred_directive(Out, _):-
  writeln(Out, ''),
  !.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Parser                            %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

parse_pred([extrn(Code)])--> parse_extrn(Code), !.
parse_pred(Pred)-->
  parse_body(period, Parsed),
  {parse_no_errors},
  {parse_rule_or_fact(Parsed, Pred)},
  !.
parse_pred([parseerror(Type, Location)], _, []):-
  retract(parse_error_location_i(Type, Location)),
  !.
parse_pred([parseerror(parse, 0)], _PredTokens, _):-
%  debug_msg(['Failed parsing predicate']),
%  debug_msg(PredTokens),
  !.

parse_rule_or_fact([prefix((:-),Directive)|Tail], [directive([Directive|Tail])]):- !.
parse_rule_or_fact([infix((:-),Head,First)|Body], [rule(Head), First|Body]):- !.
parse_rule_or_fact([infix((-->),Head,First)|Body], [dcg(Head)|NBody]):-
  parse_dcg([First|Body], NBody),
  !.
parse_rule_or_fact([infix((;),Left,Right)|Body], [rule(Head), infix((;), First, Second)|PredBody]):-
  parse_rule_or_fact([Left, Right|Body], [rule(Head), First,Second|PredBody]),
  !.
parse_rule_or_fact([X], [fact(X)]):- !.
parse_rule_or_fact(X, X):- !.

parse_dcg([], []):- !.
parse_dcg([extrn(XBody)|Body], [dcgprolog(Parsed)|NBody]):-
  parse_extrn(XBody, Parsed),
  !,
  parse_dcg(Body, NBody).
parse_dcg([comment(Comment, Term)|Body], [comment(Comment, PTerm)|NBody]):-
  parse_dcg([Term], [PTerm]),
  !,
  parse_dcg(Body, NBody).
parse_dcg([comment_line(Comment, Term)|Body], [comment_line(Comment, PTerm)|NBody]):-
  parse_dcg([Term], [PTerm]),
  !,
  parse_dcg(Body, NBody).
parse_dcg([X|Body], [X|NBody]):-
  !,
  parse_dcg(Body, NBody).

parse_extrn(XBody, Parsed):-
  atom_chars(XBody, Chars),
  append([_|NChars], [_], Chars), % Descarto los '{' '}' que envuelven el codigo prolog
  unget_chars(NChars),
  unget_char(' '), % Hack
  save_cur_line,
  tokenize_one_pred(none, Pred),
  pred_tokens(Pred, PredTokens, _),
  restore_cur_line,
  parse_body(eof, Parsed, PredTokens, []),
  !.
  
:-dynamic parse_error_location_i/2.
assert_parse_error_location(Type, Remaining, _):-
  length(Remaining, Location),
  assert_parse_error_location(Type, Location),
  !,
  fail.

assert_parse_error_location(_, Location):-
  call(parse_error_location_i(_, Loc)),
  Loc =< Location,
  !.
assert_parse_error_location(Type, Location):-
  retractif(parse_error_location_i(_, _)),
  assert(parse_error_location_i(Type, Location)),
  !.
  
parse_no_errors:-
  not(call(parse_error_location_i(_, _))),
  !.

parse_body(Finalizer, [])-->parse_token(Finalizer).
parse_body(Finalizer, Body)-->
  parse_subgoal_g(MedPrec, SubGoal1, Cont),
  parse_subgoal_g_cont(Cont, MedPrec, SubGoal1, _, SubGoal),
  parse_subgoal_cont(Finalizer, SubGoal, Body),
  !.
parse_body(Finalizer, Body)-->
  parse_cut,
  parse_subgoal_cont(Finalizer, cut, Body).

parse_subgoal(NewPrec, SubGoal)-->
  parse_subgoal(1101, NewPrec, SubGoal).

parse_subgoal(MaxPrec, NewPrec, SubGoal)-->
  parse_subgoal_g(MedPrec, SubGoal1, Cont),
  { MedPrec < MaxPrec },
  parse_subgoal_g_cont(Cont, MedPrec, SubGoal1, NewPrec, SubGoal).

parse_subgoal_cont(Finalizer, SubGoal, [SubGoal]) --> parse_token(Finalizer).
parse_subgoal_cont(Finalizer, SubGoal, [SubGoal|Body])-->
  parse_comma,
  !,
  parse_body(Finalizer, Body),
  !.
parse_subgoal_cont(Finalizer, SubGoal, [infix((;), SubGoal, Right)|Body])-->
  parse_colon,
  !,
  parse_body(Finalizer, [Right|Body]),
  !.
parse_subgoal_cont(_, _, _) --> assert_parse_error_location(syntax).

parse_subgoal_g(NewPrec, SubGoal, Cont)-->
  parse_token(openpar),
     parse_body(closepar, SubGoal1),
  parse_subgoal_op(0, SubGoal1, NewPrec, SubGoal, Cont).
parse_subgoal_g(NewPrec, SubGoal, Cont)-->
  parse_token(startsnip),
     parse_body(endsnip, SubGoal1),
  parse_subgoal_op(0, snip(SubGoal1), NewPrec, SubGoal, Cont).
parse_subgoal_g(NewPrec, SubGoal, Cont)-->
  parse_subgoal1(MedPrec, SubGoal1),
  parse_subgoal_g_cont(y, MedPrec, SubGoal1, MedPrec2, SubGoal2),
  parse_subgoal_op(MedPrec2, SubGoal2, NewPrec, SubGoal, Cont).

parse_subgoal_g_cont(y, MedPrec, SubGoal1, NewPrec, SubGoal)-->
  {parse_no_errors},
  parse_subgoal_op(MedPrec, SubGoal1, NewPrec, SubGoal, _).
parse_subgoal_g_cont(_, Prec, SubGoal, Prec, SubGoal)-->[].

parse_subgoal_op(CurPrec, Left, NewPrec, SubGoal, y)-->
  parse_infix_op(CurPrec, Left, NewPrec, SubGoal).
parse_subgoal_op(CurPrec, Left, NewPrec, SubGoal, y)-->
  parse_postfix_op(CurPrec, Left, NewPrec, SubGoal).
parse_subgoal_op(CurPrec, SubGoal, CurPrec, SubGoal, n)-->[].

parse_subgoal1(0, SubGoal)-->
  parse_functor_call(SubGoal).
parse_subgoal1(Prec, SubGoal)-->
  parse_prefix_op(Prec, SubGoal).
parse_subgoal1(0, atom(SubGoal))--> parse_atom(SubGoal).
parse_subgoal1(0, var(SubGoal))--> parse_var(SubGoal).
parse_subgoal1(0, unnamed_var(SubGoal))--> parse_unnamed_var(SubGoal).
parse_subgoal1(0, number(Number))--> parse_number(Number).
parse_subgoal1(0, list(List)) --> parse_list(List).
parse_subgoal1(0, cut)--> parse_cut.
parse_subgoal1(0, extrn(Code))--> parse_extrn(Code).
parse_subgoal1(Prec, comment(Comment, Term))--> parse_comment(Comment, Term), parse_subgoal1(Prec, Term).

parse_infix_op(CurPrec, Left, OpPrec, infix(Op, Left, Right))-->
  parse_op(Op, Assoc, OpPrec),
  { not(on_test(Op, [',', (;)])) }, % El ',' se podria quitar, el (;) es para que sea mas eficiente
  { validate_pre_op(Assoc, OpPrec, CurPrec) },
  parse_subgoal(NewPrec, Right),
  { validate_post_op(Assoc, OpPrec, NewPrec) }.

validate_pre_op(_, 0, _):- !,fail. % Operador deshabilitado
validate_pre_op(xfx, Prec, CurPrec):- CurPrec < Prec.
validate_pre_op(xfy, Prec, CurPrec):- CurPrec < Prec.
validate_pre_op(yfx, Prec, CurPrec):- CurPrec =< Prec.

validate_post_op(_, 0, _):- !,fail. % Operador deshabilitado
validate_post_op(xfx, Prec, NewPrec):- NewPrec < Prec.
validate_post_op(xfy, Prec, NewPrec):- NewPrec =< Prec.
validate_post_op(yfx, Prec, NewPrec):- NewPrec < Prec.

parse_prefix_op(NewPrec, prefix(Op, Right))-->
  parse_op(Op, Assoc, _),
  { on_test(Assoc, [fx, fy]) },
  parse_subgoal(NewPrec, Right).

parse_postfix_op(CurPrec, Left, OpPrec, postfix(Op, Left))-->
  parse_op(Op, Assoc, OpPrec),
  { on_test(Assoc, [xf, yf]) },
  { validate_postfix_op(Assoc, OpPrec, CurPrec) }.

validate_postfix_op(xf,  Prec, NewPrec):- NewPrec < Prec.
validate_postfix_op(yf,  Prec, NewPrec):- NewPrec =< Prec.

parse_token(Token)-->[Token].
parse_token(Token)-->parse_comment(_, _), [Token].

parse_comment(comment(Comment, Term), Term)-->[comment(Comment)].
parse_comment(comment_line(Comment, Term), Term)-->[comment_line(Comment)].

parse_functor_call(functor(Pred, Arity, ParamList))-->
  parse_functor(Pred, Arity, ParamList),
  !.

parse_functor(Pred, Arity, ParamList)-->
  parse_atom(Pred),
  parse_openpar,
  parse_param_list(ParamList),
  { length(ParamList, Arity) },
  !.

parse_param_list([param(Param)|ParamTail])-->
  parse_subgoal(_, Param),
  parse_param_list_cont(ParamTail).
parse_param_list([])-->parse_closepar.

parse_param_list_cont(ParamTail)-->
  parse_comma,
  parse_param_list(ParamTail).
parse_param_list_cont([])-->parse_closepar.
parse_param_list_cont([])-->assert_parse_error_location(syntax).

parse_atom(Atom)-->[atom(Atom)].
parse_atom(Atom)-->[op(Atom)].
%parse_atom('(,)')--> [openpar, comma, closepar].  % Caso especial, operador (,) llega como 'openpar comma closepar'

parse_var(Var)-->[var(Var)].

parse_unnamed_var(Var)-->[unnamed_var(Var)].

parse_number(Number)-->[number(Number)].

parse_extrn(Code)-->[extrn(Code)].

parse_list(List)-->
  parse_token(startlist),
  parse_list_item(List),
  parse_token(endlist),
  !.

parse_list_item([Param|ParamTail])-->
  parse_subgoal(_, Param),
  parse_list_item_cont(ParamTail).
parse_list_item([])-->[].
parse_list_item(_)--> assert_parse_error_location(list_syntax).

parse_list_item_cont(ParamTail)-->
  parse_comma,
  parse_list_item(ParamTail).
parse_list_item_cont([list([Param])])-->
  parse_op('|'),
  parse_subgoal(_, Param).
parse_list_item_cont([])-->[].
parse_list_item_cont(_)--> assert_parse_error_location(list_syntax).

parse_op(Op)-->parse_op(Op, _, _), !.

parse_op(Op, OpType, OpPrecedence)-->
  [op(Op)],
  {operator_type(Op, OpType, OpPrecedence)}.
parse_op(Op, OpType, OpPrecedence)-->
  parse_atom(Op),
  {operator_type(Op, OpType, OpPrecedence)}.
parse_op(',', OpType, OpPrecedence)-->
  [openpar, comma, closepar],  % Caso especial, operador (,) llega como 'openpar comma closepar'
  {operator_type(',', OpType, OpPrecedence)}.

parse_openpar-->[openpar].
parse_closepar-->[closepar].

parse_comma-->[comma].

parse_colon-->parse_op(;).

parse_cut-->[cut].

:-dynamic op_type_i/3.
add_op(Op, OpType, OpPrecedence):-
  retractif(op_type_i(Op, OpType, _)),
  assert_once(op_type_i(Op, OpType, OpPrecedence)),
  !.

operator_type(Op, OpType, OpPrecedence):-
  call(op_type_i(Op, OpType, OpPrecedence)). %Operadores definidos dinamicamente (via add_op/3)
operator_type(Op, OpType, OpPrecedence):-
  on([Op, OpType, OpPrecedence], [
                         [(:-),xfx,1200],
                         [(-->),xfx,1200],
                         [(:-),fx,1200],
                         [(?-),fx,1200],
                         [(mode),fy,1150],
                         [(public),fy,1150],
                         [(extrn),fy,1150],
                         [(visible),fy,1150],
                         [(module),fy,1150],
                         [(dynamic),fy,1150], /* SWI */
                         [(;),xfy,1100],
                         ['|',xfy,1100],
                         [',',xfy,1000],
                         [\+,fy,900], /*  SWI */
                         [->,xfy,800],
                         [\/,yfx,665],
                         [\+/,yfx,655],
                         [/\,yfx,710],
                         [=,xfy,700],
                         [is,xfx,700],
                         [=..,xfx,700],
                         [\=,xfx,700],
                         [==,xfx,700],
                         [\==,xfx,700],
                         [@<,yfx,650],
                         [@=<,yfx,650],
                         [@>,yfx,650],
                         [@>=,yfx,650],
                         [=\=,yfx,650],
                         [=:=,yfx,650],
                         [<,yfx,600],
                         [=<,yfx,600],
                         [>,yfx,600],
                         [>=,yfx,600],
                         [<<,yfx,550],
                         [>>,yfx,550],
                         [+,yfx,500],
                         [-,yfx,500],
                         [:,xfy,500],
                         [+,fx,500],
                         [-,fx,500],
                         [*,yfx,400],
                         [/,yfx,400],
                         [mod,yfx,400],
                         [//,yfx,400],
                         [^,xfy,300],
                         [*,fy,300],
                         [&,fy,300],
                         [\,fy,300],
                         [**,yfx,200]
                        ]). % No lleva cut pues se puede repetir el mismo operador

:-dynamic debug_remaining/3.
debug_remaining(Msg, X, []):-
  debug_msg([remaining, Msg, X]),
  fail.
debug_remaining(_)-->[].

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Tokenizer                         %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

token_line(In, token(LNo, Token)):-
  peek_first_char(In, Char),
  s_token(Char, In, _,  Token),
  cur_line_ack(LNo),
  token_line1(Token),
  !.

token_line1(unexpected(X)):-
  concat(['          Unexpected token: ', X], Msg),
  debug_msg(Msg),
  !.
token_line1(_):- !.

s_token(Char, In, TokenType, Token):-
  token(Char, In, TokenType, Token1),
  !,
  s_token0(In, TokenType, Token1, Token).

s_token0(In, TokenType, Token1, Token):-
  on_test(TokenType, [atom, number, var, unnamed_var]),
  peek_first_char(In, Char),
  Char = '/', % Se chequea anticipado que Char sea '/'
  Token1 =.. [Type, Arg1],
  Token =.. [Type, Arg],
  s_token1(Char, In, TokenType, Arg1, Arg),
  !.
s_token0(_, _, Token, Token):- !.

s_token1(Char, In, TokenType, Arg1, Arg):-
  Char = '/', % Puede ser el comienzo de un comentario
  get_next_char(In, NChar),
  s_token2(NChar, In, TokenType, Arg1, Arg),
  !.
s_token1(_, _, _, Arg, Arg):- !.

s_token2('*', In, TokenType, Token1, Token):-
% Comentario del tipo /**/
% Lo descarto
  consume_and_peek(In, Char),
  get_until_list(Char, In, ['*', '/'], [], _Comment1),
  get_next_char(In, NChar),
  s_token3(NChar, In, TokenType, Token1, Token),
  !.
s_token2(Char, _, _, Token, Token):-
% No es comentario del tipo /**/
  test_dec_line(Char),
  unget_char_add('/'),
  !.

s_token3(Char, In, number, XNumber, Token):-
% Asumo radix 10
  ifthenelse((atom_number(XNumber, Num), integer(Num)),
    get_number3(Char, 10, In, Number1),
    get_decimal2(Char, In, Number1)),
  atom_chars(Number, Number1),
  concat([XNumber, Number], NNumber),
  peek_first_char(In, NChar),
  !,
  s_token1(NChar, In, number, NNumber, Token).
s_token3(Char, In, TokenType, XAtom, Token):-
  get_atom_any(Char, In, Atom1),
  atom_chars(Atom, Atom1),
  concat([XAtom, Atom], NAtom),
  peek_first_char(In, NChar),
  !,
  s_token1(NChar, In, TokenType, NAtom, Token).

%%%%%%%%%%%%%
% Tokens    %
%%%%%%%%%%%%%
token(end_of_file, _, eof, eof):- !.
token(Char, In, TokenType, Token):-
  is_whitespace(Char),
  consume_and_peek(In, NChar),
  cur_line_ack(_),
  !,
  token(NChar, In, TokenType, Token).
token(Char, In, atom, atom(Atom)):-
  is_alpha_lower(Char),
  get_atom(Char, In, Atom1),
  atom_chars(Atom, Atom1),
  !.
token(Char, In, var, var(Var)):-
  is_alpha_upper(Char),
  get_atom_any(Char, In, Var1),
  atom_chars(Var, Var1),
  !.
token(Char, In, number, number(Num)):-
  get_number(Char, In, Num1),
  atom_chars(Num, Num1),
  !.
token('`', In, number, number(SChar)):-  % Codigo ASCII de 1 caracter
  consume_and_peek(In, Char),
  consume_char(In),
  concat(['0''', Char], SChar),
  !.
token('_', In, unnamed_var, unnamed_var(Var)):-
  get_atom_any('_', In, Var1),
  atom_chars(Var, Var1),
  !.
token('%', In, TokenType, Token):-
  get_until_eol(In, Comment1),
  comment_or_discard(In, comment_line, Comment1, TokenType, Token),
  !.
token('/', In, TokenType, Token):-
  % Forward slash, puede ser comment u operator
  get_next_char(In, Char),
  comment_or_operator(Char, In, TokenType, Token),
  !.
token('.', In, period, period):-
  consume_char(In),
  !.
token('!', In, TokenType, Token):-
  % ! puede ser cut, endsnip o atom
  get_next_char(In, Char),
  endsnip_or_cut_or_atom(In, Char, TokenType, Token),
  !.
token(',', In, comma, comma):-
  consume_char(In),
  !.
token(';', In, colon, op((;))):-
  consume_char(In),
  !.
token('(', In, TokenType, Token):-
  get_next_char(In, Char),
  openpar_or_operator(Char, In, TokenType, Token),
  !.
token(')', In, closepar, closepar):-
  consume_char(In),
  !.
token('[', In, TokenType, Token):-
  % LeftParen puede ser startlist o startsnip
  get_next_char(In, Char),
  startlist_or_startsnip(Char, In, TokenType, Token),
  !.
token(']', In, endlist, endlist):-
  consume_char(In),
  !.
token('''', In, string, atom(String)):-  % Considero los strings como atomos
  token_string('''', In, String),
  !.
token('"', In, string, atom(String)):-  % Considero los strings como atomos
  token_string('"', In, String),
  !.
token('$', In, string, atom(String)):-  % Considero los strings como atomos
  token_string('$', In, String),
  !.
token(Char, In, operator, op(Operator)):-
  operator(Char, In, [FirstChar|Operator1]),
  atom_chars(Operator, [FirstChar|Operator1]),
  !.
token('{', In, extrn, extrn(Code)):-  % Codigo externo { ... }
  get_until_char('{', In, '}', Code1),
  consume_char(In),
  atom_chars(Code2, Code1),
  concat([Code2, '}'], Code),
  !.
token(Char, In, unexpected, unexpected(String)):-
  get_until_eol(In, String1),
  atom_chars(String, [Char|String1]),
  !.

token_string(Delimiter, In, String):-
  token_string1(Delimiter, In, String1),
  concat([Delimiter, String1, Delimiter], String),
  !.

token_string1(Delimiter, In, String):-
  consume_and_peek(In, Char),
  get_until_char(Char, In, Delimiter, String1),
  consume_char(In),
  atom_chars(String2, String1),
  token_string2(Delimiter, In, String2, String),
  !.

token_string2(Delimiter, In, String2, String):-
  peek_first_char(In, Delimiter), % Delimitador escapeado (ej: '')
  token_string1(Delimiter, In, String1),
  concat([String2, Delimiter, Delimiter, String1], String),
  !.
token_string2(_, _, String, String):- !.

openpar_or_operator(Char, In, TokenType, Token):-
  ifthenelse(Char=',', (Operator=['(',Char,')'], consume_char(In)), operator(Char, In, Operator)),
  Operator \= [],
  s_peek_char(In, NChar),
  openpar_or_operator1(NChar, In, Operator, TokenType, Token),
  !.
openpar_or_operator(_, _, openpar, openpar):- !.

openpar_or_operator1(')', In, Operator1, operator, op(Operator)):-
  append(['('|Operator1], [')'], Operator2),
  atom_chars(Operator, Operator2),
  consume_char(In),
  !.
openpar_or_operator1(Char, _, Chars, openpar, openpar):-
% Si no era un operador, debo hacer unget de los chars consumidos
  test_dec_line(Char),
  unget_chars_add(Chars),
  !.

startlist_or_startsnip('!', In, startsnip, startsnip):-
  consume_char(In),
  !.
startlist_or_startsnip(_, _, startlist, startlist):- !.

endsnip_or_cut_or_atom(In, ']', endsnip, endsnip):-
  consume_char(In),
  !.
endsnip_or_cut_or_atom(_, '/', atom, atom(!)):- !.
endsnip_or_cut_or_atom(_, _, cut, cut):- !.

comment_or_operator('*', In, TokenType, Token):-
  !,
  get_next_char(In, Char),
  get_until_list(Char, In, ['*', '/'], [], Comment1),
  comment_or_discard(In, comment, Comment1, TokenType, Token),
  !.
comment_or_operator(Char, In, operator, op(Operator)):-
  operator(Char, In, Operator1),
  atom_chars(Operator, ['/'|Operator1]),
  !.

comment_or_discard(In, _, _, TokenType, Token):-
  discard_comments,
  !,
  cur_line_ack(_),
  ifthenelse(get_next_char(In, Char), token(Char, In, TokenType, Token), (TokenType=eof, Token=eof)),
  !.
comment_or_discard(_, comment_line, Comment1, comment_line, comment_line(Comment)):-
  atom_chars(Comment, Comment1),
  !.
comment_or_discard(In, comment, Comment1, comment, comment(Comment)):-
  append(Comment2, [_], Comment1),
  atom_chars(Comment, Comment2),
  consume_char(In),
  !.

operator(Char, In, [Char|Operator]):-
  on_test(Char, [':', '-', '+', '*', '/0', '<','>', '=', '~', '\\', '/', '|', '@', '?', '^', '&', '.']),
  get_next_char(In, NChar),
  !,
  operator(NChar, In, Operator).
operator(_, _, []):- !.

get_atom(Char, In, [Char|Atom]):-
  is_alpha_lower(Char),
  get_next_char(In, NChar),
  !,
  get_atom_any(NChar, In, Atom).

get_atom_any(Char, In, [Char|Atom]):-
  is_alpha_or_digit(Char),
  get_next_char(In, NChar),
  !,
  get_atom_any(NChar, In, Atom).
get_atom_any(_, _, []):- !.

get_number(Char, In, [Char|Num]):-
  is_digit(Char),
  get_next_char(In, NChar),
  !,
  get_number1(Char, NChar, In, Num).

get_number1(RadixChar, '''', In, [''''|Num]):-
% Caso numero con radix o codigo ascii
  ifthenelse(RadixChar='0', Radix=10, (char_code(RadixChar, Radix1), Radix is Radix1 - 48)),
  get_next_char(In, NChar),
  !,
  get_number2(NChar, Radix, In, Num).
get_number1(_, Char, In, Num):-
  !,
  get_number3(Char, 10, In, Num).

get_number2(Char, 10, In, [Char]):-
% Caso codigo ascii, ej: 0'a
  consume_char(In),
  !.
get_number2(Char, Radix, In, Num):-
  !,
  get_number3(Char, Radix, In, Num).

get_number3(Char, Radix, In, [Char|Num]):-
  is_digit(Char),
  char_code(Char, DChar),
  DChar =< Radix + 48,
  get_next_char(In, NChar),
  !,
  get_number3(NChar, Radix, In, Num).
get_number3('.', 10, In, Decimals):-
% Caso el numero es un float, OJO porque puede que no sea decimal sino que justo termina el predicado.
  get_next_char(In, NChar),
  !,
  get_decimal1(NChar, In, Decimals).
get_number3(_, _, _, []):- !.

get_decimal1(Char, In, ['.', Char|Decimals]):-
% Si efectivamente es un decimal
  is_digit(Char),
  get_next_char(In, NChar),
  !,
  get_decimal2(NChar, In, Decimals).
get_decimal1(Char, _, []):-
% Si no era un decimal, entonces debemos hacer un unget del '.'
  test_dec_line(Char),
  unget_char_add('.'),
  !.

get_decimal2(Char, In, [Char|Decimals]):-
  is_digit(Char),
  get_next_char(In, NChar),
  !,
  get_decimal2(NChar, In, Decimals).
get_decimal2(_, _, []):- !.

get_until_eol(In, String):-
  get_next_char(In, Char),
  get_until_char(Char, In, '\n', String),
  !.

get_until_char(end_of_file, _, _, []):- !.
get_until_char('\\', In, SChar, ['\\',Char|String]):-
% escaping
  backslash_is_escape_char,
  get_next_char(In, Char),
  get_next_char(In, NChar),
  !,
  get_until_char(NChar, In, SChar, String).
get_until_char(Char, _, Char, []):- !.
get_until_char(Char, In, SChar, [Char|String]):-
  get_next_char(In, NChar),
  !,
  get_until_char(NChar, In, SChar, String).

get_until_list(Char, _, [Char], _, [Char]):- !.
get_until_list(Char, In, [Char|List], IString, String):-
  get_next_char(In, NChar),
  !,
  get_until_list(NChar, In, List, [Char|IString], String).
get_until_list(Char, In, List, [], [Char|String]):-
  get_next_char(In, NChar),
  !,
  get_until_list(NChar, In, List, [], String).
get_until_list(Char, In, List, IString1, String):-
  reverse(IString1, IString),
  append(IString, MString, String),
  append(IString, List, NList),
  !,
  get_until_list(Char, In, NList, [], MString).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Scanner                          %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
peek_first_char(In, Char):-
  s_peek_first_char(In, Char),
  !.

get_next_char(In, Char):-
  s_get_char(In, _),
  s_peek_char(In, Char),
  ifthen(Char='\n', inc_line),
  !,
  Char \= end_of_file.

consume_char(In):-
  s_get_char(In, _),
  s_peek_char(In, Char),
  ifthen(Char='\n', inc_line),
  !.

consume_and_peek(In, NChar):-
  get_next_char(In, NChar),
  !.
consume_and_peek(_, end_of_file):- !.

:-dynamic next_char_i/1.
s_at_end_of_stream(_):-
  call(next_char_i(_)),
  !,
  fail.
s_at_end_of_stream(none):- !.
s_at_end_of_stream(Stream):-
  at_end_of_stream(Stream),
  !.

unget_char_add(Char):-
  asserta(next_char_i(Char)),
  test_dec_line(Char),
  !.

unget_char(Char):-
  assert(next_char_i(Char)),
  test_dec_line(Char),
  !.
  
test_dec_line('\n'):-
  dec_line,
  !.
test_dec_line(_):- !.

s_clear_unget_buffer:-
  retractall(next_char_i(_)),
  !.

unget_chars([]):- !.
unget_chars([Char|Chars]):-
  unget_char(Char),
  !,
  unget_chars(Chars).

unget_chars_add(Chars):- 
  reverse(Chars, RChars),
  unget_chars_add1(RChars),
  !.

unget_chars_add1([]):- !.
unget_chars_add1([Char|Chars]):-
  unget_char_add(Char),
  !,
  unget_chars_add1(Chars).

s_peek_first_char(_, Char):-
  call(next_char_i(XChar)),
  !,
  Char = XChar.
s_peek_first_char(none, Char):-
  !,
  Char = end_of_file.
s_peek_first_char(In, Char):-
  peek_char(In, Char),
  !.

s_peek_char(_, Char):-
  call(next_char_i(XChar)),
  !,
  Char = XChar.
s_peek_char(none, Char):-
  !,
  Char = end_of_file.
s_peek_char(In, Char):-
  peek_char(In, Char),
  !.

s_get_char(_, Char):-
  retract(next_char_i(XChar)),
  !,
  Char = XChar.
s_get_char(none, Char):-
  !,
  Char = end_of_file.
s_get_char(In, Char):-
  get_char(In, Char),
  !.

is_alpha_lower(Char):-
  char_code(Char, Code),
  Code >= 0'a, Code =< 0'z,
  !.

is_alpha_upper(Char):-
  char_code(Char, Code),
  Code >= 0'A, Code =< 0'Z,
  !.

is_alpha('_'):- !.
is_alpha(Char):-
  char_code(Char, Code),
  ((Code >= 0'A, Code =< 0'Z) ; (Code >= 0'a, Code =< 0'z)),
  !.

is_alpha_or_digit(Char):-
  is_alpha(Char),
  !.
is_alpha_or_digit(Char):-
  is_digit(Char),
  !.

is_whitespace(Char):-
  char_code(Char, Code),
  Code =< 32,
  !.

initialize_line_count:-
  nb_setval(ln, 1),
  nb_setval(nln, 1),
  !.

inc_line:-
  nb_getval(nln, Cur),
  Cur1 is Cur+1,
  nb_setval(nln, Cur1),
  !.
  
dec_line:-
  nb_getval(nln, Cur),
  Cur1 is Cur-1,
  nb_setval(nln, Cur1),
  !.

cur_line_ack(LNo):-
  nb_getval(nln, NLNo),
  nb_getval(ln, LNo),
  nb_setval(ln, NLNo),
  !.

:-dynamic saved_cur_line_i/2.
save_cur_line:-
  nb_getval(nln, NLNo),
  nb_getval(ln, LNo),
  assert(saved_cur_line_i(LNo, NLNo)),
  !.

restore_cur_line:-
  retract(saved_cur_line_i(LNo, NLNo)),
  nb_setval(nln, NLNo),
  nb_setval(ln, LNo),
  !.
  
push_line_number(File):-
  assert(script_file_i(File)),
  save_cur_line,
  initialize_line_count,
  !.
  
pop_line_number:-
  retract(script_file_i(_)),
  !,
  restore_cur_line,
  !.
  
get_current_file(File):-
  call(script_file_i(File)),
  !.
get_current_file([]):- !.


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% SICSTUS - compatibility pack ;) %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
/*
:-dynamic runtime_entry/1.
runtime_entry(start):-
 main,
 halt.
 
writeln(Msg):-
  current_output(Out),
  writeln(Out, Msg),
  !.
  
nb_setval(Id, Value):-
  retractif(nb_setval_i(Id, _)),
  assert(nb_setval_i(Id, Value)),
  !.

nb_getval(Id, Value):-
  call(nb_setval_i(Id, Value)),
  !.
  
succ(A,B):-
  B is A+1,
  !.
  
is_digit(Char):-
  char_code(Char, Code),
  Code >= 0'0,
  Code =< 0'9,
  !.
  
atom_number(Atom, Number):-
  number(Number),
  number_chars(Number,L),
  atom_chars(Atom,L),
  !.
atom_number(Atom, Number):-
  atom(Atom),
  atom_chars(Atom,L),
  number_chars(Number,L),
  !.
  
not(Goal):-
  \+ Goal.
  
reverse(A, B) :- 
  !,
  reverse(A, [], B).

reverse([], New, New) :- !.
reverse([X|L], Old, New) :-
  !,
reverse(L, [X|Old], New).  
        
term_to_atom([Head|Tail], Y):-
  term_to_atom1([Head|Tail], X1),
  term_to_atom2('.', X1, Y),
  !.  
term_to_atom(X, Y):-
  X =.. [Functor,First|L],
  term_to_atom1([First|L], X1),
  term_to_atom2(Functor, X1, Y),
  !.
term_to_atom(X, X):- !.

term_to_atom1([], []):- !.
term_to_atom1([X|Tail], [Y|YTail]):- 
  term_to_atom(X, Y),  
  !,
  term_to_atom1(Tail, YTail).  
  
term_to_atom2('.', X1, Y):-
  concat_c(X1, ', ', X2),
  concat(X2, X3),
  concat(['[', X3, ']'], Y),
  !.
term_to_atom2(Op, [A,B], Y):-
  operator_type(Op, OpType, _),
  on_test(OpType, [xfx, xfy, yfx]),
  concat([A,Op,B], Y),
  !.  
term_to_atom2(Op, [A], Y):-
  operator_type(Op, OpType, _),
  on_test(OpType, [xf, yf]),
  concat([A,Op], Y),
  !.  
term_to_atom2(Op, [A], Y):-
  operator_type(Op, OpType, _),
  on_test(OpType, [xf, yf]),
  concat([Op,A], Y),
  !.  
term_to_atom2(Functor, X1, Y):-
  concat_c(X1, ', ', X2),
  concat(X2, X3),
  concat([Functor, '(', X3, ')'], Y),
  !.  

file_name_extension(Base, Ext, File):- 
  atom_chars(File, CFile),
  reverse(CFile, CFile1),
  append(CRExt, ['.'|CRBase], CFile1),
  reverse(CRExt, CExt),
  reverse(CRBase, CBase),
  atom_chars(Ext, CExt),
  atom_chars(Base, CBase),
  !.
file_name_extension(X, '', X):- !.

downcase_atom(Arg, LArg):-
  atom_chars(Arg, CArg),
  downcase_atom1(CArg, CLArg),
  atom_chars(LArg, CLArg),
  !.
  
downcase_atom1([], []):- !.
downcase_atom1([UChar|Chars], [LChar|LChars]):- 
  is_alpha_upper(UChar),
  char_code(UChar, CUChar),
  CLChar is CUChar + (0'a - 0'A),
  char_code(LChar, CLChar),
  !,
  downcase_atom1(Chars, LChars).
downcase_atom1([Char|Chars], [Char|LChars]):- 
  !,
  downcase_atom1(Chars, LChars).
  

is_sicstus:- !.
*/
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% FIN Sicstus compatibility pack %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
is_sicstus:- !,fail.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Auxiliares                     %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
  
file_base_name_and_path(File, Path, FileName):- 
  atom_chars(File, CFile),
  reverse(CFile, CFile1),
  append(CRFileName, ['/'|CRPath], CFile1),
  reverse(CRFileName, CFileName),
  atom_chars(FileName, CFileName),
  reverse(CRPath, CPath),
  atom_chars(Path, CPath),
  !.
file_base_name_and_path(File, '', File):- !.

compose_file(FileName, File):-
	atom_chars(FileName, [''''|FileName1]),
	append(FileName2, [''''], FileName1),
	atom_chars(File, FileName2),
	!.
compose_file(FileName, File):-
  file_name_extension(FileBase, FileExt, FileName),
  file_base_name_and_path(FileBase, FilePath, RFileName),
  ifthenelse(FilePath='', default_path(FPath), FPath=FilePath),
  ifthenelse(FileExt='', default_extension(FExt), FExt=FileExt),
  compose_file1(FPath, FExt, RFileName, File),
  !.

compose_file1('', FExt, RFileName, File):-
  compose_file2(FExt, RFileName, File),
  !.
compose_file1(FPath, FExt, RFileName, File):-
  compose_file2(FExt, RFileName, File1),
  concat([FPath, '/', File1], File),
  !.

compose_file2('', File, File):- !.
compose_file2(Ext, FileBase, File):-
  concat([FileBase, '.', Ext], File),
  !.

on_test(X,[X|_]) :- !.
on_test(X,[_|Tail]) :-
  !,
  on_test(X,Tail).

ifthenelse(X,Y,Z):-
  X->Y;Z.

ifthen(X,Y):-
  X->!,Y.
ifthen(_,_):-!.

concat([], []):- !.
concat([A|Tail], Y):-
% Sicstus hack
  number(A),
  atom_number(B,A),
  concat([B|Tail], Y).
concat([X], X):- !.
concat([A|Tail], Y):-
  concat(Tail, X),
  atom_concat(A,X,Y),
  !.

on(X,[X|_]).
on(X,[_|Tail]) :-
  on(X,Tail).

conj(Bag, Set) :-
  conj1(Bag,[], Set),
    !.

conj1([],_,[]) :- !.
conj1([X| Tail], Rep, Tail1) :-
  on_test(X, Rep) ,
  !,
  conj1(Tail, Rep, Tail1).
conj1([X| Tail], Rep, [X| Tail1]) :-
  !,
  conj1(Tail, [X| Rep], Tail1) .

intersec(_, [],[]) :- !.
intersec([],_,[]) :- !.
intersec([X|Tail],Y,[X|Tail1]) :-
  on_test(X,Y),
  !,
  intersec(Tail,Y,Tail1).
intersec([_|Tail],Y,Tail1) :-
    !,
  intersec(Tail,Y,Tail1).

concat_c([],_,[]) :- !.
concat_c([X],_,[X]) :- !.
concat_c([X|Tail],Oper,[X,Oper|Tail1]) :-
  !,
  concat_c(Tail,Oper,Tail1) .

assert_once(X):-
  call(X),
  !.
assert_once(X):-
  assert(X),
  !.
  
retractif(X):-
  retract(X),
  !.
retractif(_):- !.

debug_msg(X):-
  writeln(X),
  !.

debug_msg_pred(Pred):-
  debug_msg_pred(Pred, _),
  !.

debug_msg_pred([], _):-
  writeln(''),
  !.
debug_msg_pred([token(CurLine, X)|Y], CurLine):-
  write(X),
  write(' '),
  !,
  debug_msg_pred(Y, CurLine).
debug_msg_pred([token(CurLine, X)|Y], _):-
  writeln(''),
  write(X),
  write(' '),
  !,
  debug_msg_pred(Y, CurLine).

writeln(Stream, Msg):-
  write(Stream, Msg),
  nl(Stream),
  !.
