require 'rubygems'
require 'jira'
require 'httparty'


class JiraClient
  include HTTParty

  # Constructor de la clase
  # Parametros: username: Usuario para autenticarse a Jira
  # 			  		password: password utilizada junto con el username para autenticarse con Jira
  # 			  		site: URL de Jira a la cual quiere autenticarse
  # Sete las variables a utilizar en la clase

  # Constructor for the class
  # Parameters: username: Username to authenticate Jira
  # 						Password: password used with the username to authenticate Jira
  # 						Site: Jira URL to which you want to authenticate
  # Sets the class variables to use
  def initialize(username, password, site)
    @user = username
    @pass = password
    @url = site
    self.class.base_uri site
    self.class.headers 'Content-Type' => 'application/json', 'Accept' => 'application/json'
    self.class.basic_auth username, password
  end

  # Parametros: No tiene
  # Retorna un mensaje de error en caso de fallo de la autenticacion
  # Contenido: Realiza la autenticacion con la herramienta de gestion JIRA

  # Parameters: None
  # Returns an error message if authentication failure
  # Content: Does the authentication with JIRA
  def doAuthentication()
    options = {
        username: @user,
        password: @pass,
        site: @url,
        context_path: '',
        auth_type: :basic
    }
    @client = JIRA::Client.new(options)
    message = chkAuthentication()
    return message
  end

  # Parametros: No tiene
  # Retorna todos los proyectos asociados al @client autenticado previamente

  # Parameters: None
  # Returns all projects associated with previously authenticated @client
  def getProjects()
    @client.Project.all
  end

  # Parametros: No tiene
  # Retorna todas las Tareas (issues or tasks) asociadas que tiene el @client autenticado previamente

  # Parameters: None
  # Returns all tasks (issues) that has associated the @client previously authenticated
  def getUserIssues()
    @client.Issue.all
  end

  # Parametros: issue_id: Es el id de una issue (Tarea) en la herramienta de gestion Jira
  # Retorna la tarea identificada por el parametro issue_id o un mensaje de error si no existe la tarea.

  # Parameters: issue_id: The id of an issue in Jira
  # Returns the task identified by the issue_id or an error message if the task does not exist.
  def getIssue(issue_id)
    message = ''
    begin
      issue = @client.Issue.find(issue_id.to_s)
    rescue JIRA::HTTPError
      message = 'Task with ID: ' + issue_id.to_s + ' does not exist.'
    end
    return issue, message
  end

  # Parametros: issue_id: ID de la tarea (issue) a la cual se le quiere cargar horas
  #             comment: Comentario que se quiera asociar a la carga de horas
  #             date: Fecha en la cual se realiza la carga de horas
  #             timeSpent: Tiempo dedicado que se desea cargar a la tarea (HH:mm)
  # Retorna un mensaje si el worklog(Carga de Horas) creado o modificado en la tarea especificada por issue_id es exitoso o no

  # Parameters: issue_id: Task ID (issue) that you want to load hours
  # 						Comment: Comment you want to associate with the worklog
  # 						Date: Date worklog
  # 						Timespent: Time spent in the Task (issue) specified by issue_id (HH:mm)
  # Returns a message if the worklog created or modified in the task specified by issue_id is successful or not
  def createWorklog(issue_id, comment, date, time_Spent)
    issue, message = self.getIssue(issue_id)

    if message.empty?
      time_Spent_worklog = getTimeInCorrectFormat(time_Spent)
      date_worklog = getDateInCorrectFormat(date)
      new_Worklog = issue.worklogs.build
      new_Worklog.save(comment: comment, started: date_worklog, timeSpent: time_Spent_worklog)
      message = 'Successful entry.'
    end

    return message
  end

  # Parametros: task: una instancia de una task de la base de datos
  # Realiza el impacto del cambio de datos de la tarea en Jira
  def editJiraIssue(task)
    transition_id = 0
    issue,message = getIssue(task.externalId.to_s)
    estimation = getTimeInCorrectFormat(task.estimation)
    issue.save({'fields' => {'summary' => task.name.to_s, 'description' => task.description.to_s, 'duedate' => task.due_date.to_s,
                             'priority' => {'name' => task.priority.to_s}, 'timetracking'=>{'remainingEstimate'=>estimation}}})

    transitions = getTransitions(task.externalId.to_s)
    change_status = false
    transitions['transitions'].each do |transition|
      if transition['to']['name'].to_s == task.status.to_s
        transition_id = transition['id']
        change_status = true
      end
    end
    if change_status
      doTransition(task.externalId.to_s,transition_id)
    end
    return message
  end

  # Parametros: task_id: externalId de una instancia de una Task de la base de datos
  # Busca las transiciones existentes para el estado actual de la tarea y guarda los nombres
  # de los posibles estados a los que puede cambiar la Task.
  def getNextStatus(task_id)
    transitions = getTransitions(task_id.to_s)
    names_status = transitions['transitions'].first['to']['name'].to_s
    transitions['transitions'].each do |transition|
      if transitions['transitions'].first != transition
        names_status = names_status + ';' + transition['to']['name'].to_s
      end
    end
    return names_status
  end

  # Parametros: No tiene
  # Obtiene un arreglo con todos los nombres de las prioridades existentes en el Jira utilizado.
  def getPriorities()
    priorities = getAllJiraPriorities()
    priority_names = priorities.first['name'].to_s
    priorities.each do |priority|
      if priority != priorities.first
        priority_names = priority_names + ';' + priority['name'].to_s
      end
    end
    return priority_names
  end

  private

  # Parametros: No tiene
  # Retorna un string que nos muestra un mensaje en caso de que falle la autenticacion o vacio si la autenticacion fue con exito
  # Contenido: Se invoca a una de las funciones de la clase y las sentencias rescue capturan las diferentes exceptions
  #            si las hay guardando el correspondiente mensaje de error

  # Parameters: None
  # Returns a string that shows a message in case of failure of authentication or empty if the authentication was successfully
  # Content: It calls one of the functions of the class and rescue statements capture the different exceptions and saving the corresponding error message
  def chkAuthentication()

    ok = false
    message = ""
    begin
      issues_aux = self.getUserIssues
    rescue SocketError
      message = 'URL incorrect.'
    rescue JIRA::HTTPError
      message = 'User or Password incorrect.'
    rescue Errno::ECONNREFUSED
      message =  'URL incorrect.'
    rescue Exception
      message =  'Error validating credentials.'
    end
    return message
  end


  # Parametros: time_spent: String de la forma 'HH:mm'
  # Retorna un string resultado de la conversion de time_spent en minutos concatenando una 'm' al final

  # Parameters: time_spent: String of the form 'HH: mm'
  # Returns a string conversion result of time_spent in minutes, concatenating 'm' at the end
  def getTimeInCorrectFormat(time_spent)
    aux_time = time_spent.split(':')
    resultado = (aux_time[0].to_i * 60) + aux_time[1].to_i
    return resultado.to_s + 'm'
  end

  # Parametros: date: Fecha ingresada en el FrontEnd por el usuario
  # Verifica que la fecha pasada por parametro sea igual a la actual (formato aaaa-mm--dd)
  # Si es la fecha actual, la convierte en el formato pedido por Jira
  # Si no es la actual, a date se le concatena una hora por defecto y zona horaria por defecto quedando en el formato pedido por Jira
  # Retorna la fecha en el formato que solicita Jira

  # Parameters: date: Date put by user in FrontEnd
  # Check if date equals to actual date(format aaaa-mm-dd)
  # If it's actual date, convert it in the format being requested by Jira
  # Otherwise, date is concatenated with a default time and default time zone in the format being requested by Jira
  # Returns date in the format being requested by Jira
  def getDateInCorrectFormat(date)
    time_actual = Time.new()

    if date.to_s == time_actual.strftime('%Y-%m-%d')
      date_auxiliar = time_actual.strftime('%Y-%m-%dT%H:%M:%S.000%z')
    else
      date_auxiliar = date.to_s + 'T12:00:00.000-0300'
    end

    return date_auxiliar
  end

  # Devuelve las transiciones validas para el issue con issue_id
  def getTransitions(issue_id)
    self.class.get("/rest/api/2/issue/#{issue_id}/transitions")
  end

  # Ejecuta la transicion con identificador transition_id para el issue identificado por issue_id
  def doTransition(issue_id,transition_id)
    self.class.post("/rest/api/2/issue/#{issue_id}/transitions", :body => { :transition => { :id => transition_id}}.to_json)
  end

  # Obtiene un Json que contiene todas los datos de las prioridades existentes en el Jira utilizado
  def getAllJiraPriorities()
    self.class.get("/rest/api/2/priority")
  end

end