require_relative 'JiraClient'
require_relative 'RedMineClient'

class ToolAdapter
  attr_accessor :tool_name, :client, :user_id,:tool_auth
  HALF_HOUR = 3.0
  HALF_HOUR_REDMINE = 5.0

  # Parametros: user_id: Id del User que se desea autenticar contra una herramienta de gestión
  def initialize(user_id, tool_auth = nil)
    @user_id = user_id
    @tool_auth = tool_auth
  end


  def check_credentials(tool_options)
    new_tool_auth = ToolAuth.new(tool_options)

    url = new_tool_auth.url
    user = new_tool_auth.tool_user
    password = new_tool_auth.password
    api_key = new_tool_auth.api_key
    tool_name = new_tool_auth.tool_name

    case tool_name
      when 'jira'
        @client = JiraClient.new(user,password,url)
        message = @client.doAuthentication()
      when 'redmine'
        @client = RedMineClient.new(api_key,url)
        message = @client.doAuthentication()
      else
        message = 'Error: Unknown tool'
    end
  end

  # Retorna un mensaje de error en caso de fallo de la autenticacion y una flag de exito o no
  # Contenido: Realiza la autenticacion con la herramienta de gestion
  def authenticate()


    return 'Error, credenciales invalidas' if @tool_auth.nil?

    message = ''

    @tool_name = @tool_auth.tool_name

    case @tool_name
      when 'jira'
        @client = JiraClient.new(@tool_auth.tool_user, @tool_auth.password,@tool_auth.url)
        message = @client.doAuthentication()
      when 'redmine'
        @client = RedMineClient.new(@tool_auth.api_key, @tool_auth.url)
        message = @client.doAuthentication()
      else
        message = 'Error, herramienta desconocida'
    end

  end



  # Retorna todos los Projects asociados al @client autenticado previamente
  def get_projects()
    message = self.authenticate()
    case @tool_name
      when 'jira'
        return get_projects_jira()
      when 'redmine'
        return get_projects_redmine()
      else
        return []
    end
  end


  # Retorna todas las Tasks asociadas que tiene el @client autenticado previamente
  def get_user_tasks()
    response = self.authenticate()
    raise "Error: Can't authenticate" if not response.empty?
  rescue
    return response
  else

    case @tool_name
      when 'jira'
        return get_user_tasks_jira()
      when 'redmine'
        return get_user_tasks_redmine()
      else
        return []
    end
  end

  # Parametros: task_id: Es el id de una Task en la herramienta de gestion
  #             user_id: Es el id del User para el que se requiere la tarea
  # Retorna la tarea identificada por el parametro task_id o un mensaje de error si no existe la tarea.
  def get_task(task_id)
    message = self.authenticate()

    case @tool_name
      when 'jira'
        return get_task_jira(task_id)
      when 'redmine'
        return get_task_redmine(task_id)
      else
        return []
    end
  end

  # Parametros: task: una instancia de una task de la base de datos
  # Realiza el impacto del cambio de datos de la tarea en la herramienta de gestión
  def update_task(task)
    response = self.authenticate()
    raise "Error: Can't authenticate" if not response.empty?
  rescue
    return response
  else

    case @tool_name
      when 'jira'
        return @client.editJiraIssue(task)
      when 'redmine'
        return @client.editRedMineIssue(task)
      else
        return []
    end

  end

  # Parametros: task_id: ID de la Task a la cual se le quiere cargar horas
  #             comment: Comentario que se quiera asociar a la carga de horas
  #             date: Fecha en la cual se realiza la carga de horas
  #             timeSpent: Tiempo dedicado que se desea cargar a la tarea
  #             user_id: Usuario que realiza la carga de horas
  # Retorna un mensaje si el worklog(Carga de Horas) creado o modificado en la tarea especificada por task_id es exitoso o no
  def create_entry(external_task_id, comment, date, time_spent)
    response = self.authenticate()
    raise "Error: Can't authenticate" if not response.empty?
  rescue
    return response
  else

    case @tool_name
      when 'jira'
        return create_entry_jira(external_task_id, comment, date, time_spent)
      when 'redmine'
        return create_entry_redmine(external_task_id, comment, date, time_spent)
      else
        return []
    end

  end

  # Parametros: task_id: external_id de una tarea de la base de datos
  # Retorna un arreglo que contiene los nombres de los posibles estados a los que puede pasar esa
  # tarea dependiendo del estado actual de la misma
  def get_Next_Status(task_id)
    response = self.authenticate()
    raise "Error: Can't authenticate" if not response.empty?
  rescue
    return response
  else

    case @tool_name
      when 'jira'
        return @client.getNextStatus(task_id)
      when 'redmine'
        return @client.getNextStatus(task_id)
      else
        return []
    end

  end

  # Parametros: No tiene
  # Retorna un arreglo que contiene los nombres de todas las prioridades existentes en Jira o Redmine
  def get_Priorities()
    response = self.authenticate()
    raise "Error: Can't authenticate" if not response.empty?
  rescue
    return response
  else

    case @tool_name
      when 'jira'
        return @client.getPriorities()
      when 'redmine'
        return @client.getPriorities()
      else
        return []
    end

  end

  private


  #
  # Traer tareas
  #
  #  #TODO manejo de errores
  def get_user_tasks_jira()
    message = self.authenticate()
    tasks = []
    if message.empty?
      issues = @client.getUserIssues()

      issues.each do |issue|
        task = Task.new
        task.name = issue.fields['summary'].to_s
        task.status = issue.fields['status']['name'].to_s
        task.externalId = issue.id
        task.description = issue.fields['description'].to_s
        task.due_date = issue.fields['duedate'].to_date if issue.fields['duedate']
        task.estimation = convert_estimation_Jira(issue.fields["timeoriginalestimate"])
        task.user_id = @user_id
        task.external_project_id = issue.fields['project']['id'].to_s if issue.fields['project']['id']
        task.priority =  issue.fields['priority']['name'].to_s  if issue.fields['priority']
        task.priorities = self.get_Priorities()
        task.next_status = self.get_Next_Status(issue.id)
        task.assignee = issue.fields['assignee']['displayName'].to_s if issue.fields['assignee']
        tasks.push task
      end
    end

    return tasks
  rescue
    return []
  end

  def convert_estimation_Jira(task_estimation)
    if task_estimation.nil?
      resultado = '-'
    else
      mins = ((task_estimation.to_i/60)%60)
      hours = (task_estimation.to_i/3600)
      resultado = hours.to_s + ':' + format('%02d', mins)
    end
    return resultado
  end

  def get_user_tasks_redmine()
    message = self.authenticate()
    tasks = []
    if message.empty?
      issues = @client.getIssuesUser()
      issues.each do |issue|
        task = Task.new
        task.name = issue['subject']
        task.status = issue['status']['name']
        task.externalId = issue['id'].to_s  if issue['id']
        task.description = issue['description']
        task.estimation = convert_estimation_Redmine(issue['estimated_hours'].to_s)
        task.due_date = issue['due_date'].to_date if issue['due_date']
        task.user_id = @user_id
        task.external_project_id = issue['project']['id'].to_s if issue['project']['id']
        task.priority = issue['priority']['name'] if issue['priority']
        task.priorities = self.get_Priorities()
        task.next_status = self.get_Next_Status(issue['id'])
        task.assignee = issue['assigned_to']['name'] if issue['assigned_to']
        tasks.push task

      end
    end

    return tasks

  end

  def convert_estimation_Redmine(task_estimation)
    if task_estimation.empty?
      resultado = '-'
    else
      estimation = task_estimation.to_s.split('.')
      estimation[1] = estimation[1] + '0' if estimation[1].length == 1
      if estimation[1] != nil
          aux_minutes = (estimation[1].to_i * (HALF_HOUR / HALF_HOUR_REDMINE)).to_i
      else
        aux_minutes = 0
      end
      resultado = estimation[0].to_s + ':' + format('%02d', aux_minutes)
    end
    return resultado
  end

  #
  # Traer projects
  # TODO: traer el json y parsearlo a project. No se si se usara esta funcion alguna vez
  def get_projects_jira()
    message = self.authenticate()

    projects_json = @client.getProjects()
    projects = []

    #return [] if projects_json.blank?

    projects_json.each do |project_json|
      project = Project.new
      project.name = project_json.name
      project.externalId = project_json.id

      projects.push project
    end

    return projects
  end

  def get_projects_redmine()
    message = self.authenticate()

    projects_json = @client.getProjects
    projects = []

    #return [] if projects_json.blank?

    projects_json.each do |project_json|
      project = Project.new
      project.name = project_json['project']['name']
      project.externalId = project_json['project']['id']
      projects.push project
    end

    return projects

  end

  #
  # Crear entry
  #
  def create_entry_jira(task_external_id, comment, date, time_spent)
    #TODO no hay que poner la categoria?
    return @client.createWorklog(task_external_id, comment, date, time_spent)
  end

  def create_entry_redmine(task_external_id, comment, date, time_spent)
    activity_id = '0' # esto seria la category?
    return @client.create_Time_Entry(task_external_id, date, time_spent, activity_id, comment)
  end

  #
  # Traer task en particular
  # TODO: traer el json y parsearlo a task. No se si se usara esta funcion alguna vez
  def get_task_jira(task_id)

  end

  def get_task_redmine(task_id)

  end
end
