package curtidor.utils;

import java.util.ArrayList;

import lejos.robotics.RegulatedMotor;
import lejos.robotics.RegulatedMotorListener;
import lejos.robotics.navigation.ArcRotateMoveController;
import lejos.robotics.navigation.Move;
import lejos.robotics.navigation.MoveListener;
import lejos.robotics.navigation.MoveProvider;
import lejos.util.Delay;

public class SyncDiffPilot implements RegulatedMotorListener, ArcRotateMoveController {

	private int unreportedMoves;
	private Object syncReports;

	/**
	 * Allocates a DifferentialPilot object, and sets the physical parameters of the
	 * NXT robot.<br>
	 *
	 * @param leftWheelDiameter
	 *            Diameter of the left wheel, in any convenient units (diameter
	 *            in mm is usually printed on the tire).
	 * @param rightWheelDiameter
	 *            Diameter of the right wheel. You can actually fit
	 *            intentionally wheels with different size to your robot. If you
	 *            fitted wheels with the same size, but your robot is not going
	 *            straight, try swapping the wheels and see if it deviates into
	 *            the other direction. That would indicate a small difference in
	 *            wheel size. Adjust wheel size accordingly. The minimum change
	 *            in wheel size which will actually have an effect is given by
	 *            minChange = A*wheelDiameter*wheelDiameter/(1-(A*wheelDiameter)
	 *            where A = PI/(moveSpeed*360). Thus for a moveSpeed of 25
	 *            cm/second and a wheelDiameter of 5,5 cm the minChange is about
	 *            0,01058 cm. The reason for this is, that different while sizes
	 *            will result in different motor speed. And that is given as an
	 *            integer in degree per second.
	 * @param trackWidth
	 *            Distance between center of right tire and center of left tire,
	 *            in same units as wheelDiameter.
	 * @param leftMotor
	 *            The left Motor (e.g., Motor.C).
	 * @param rightMotor
	 *            The right Motor (e.g., Motor.A).
	 * @param reverse
	 *            If true, the NXT robot moves forward when the motors are
	 *            running backward.
	 */
	public SyncDiffPilot(final double leftWheelDiameter,
			final double rightWheelDiameter, final double trackWidth,
			final RegulatedMotor leftMotor, final RegulatedMotor rightMotor,
			final boolean reverse)
	{
		_left = leftMotor;
		_left.addListener(this);
		_leftWheelDiameter = (float)leftWheelDiameter;
		_leftTurnRatio = (float)(trackWidth / leftWheelDiameter);
		_leftDegPerDistance = (float) (360 / (Math.PI * leftWheelDiameter));
		// right
		_right = rightMotor;
		_right.addListener(this);
		_rightWheelDiameter = (float)rightWheelDiameter;
		_rightTurnRatio = (float)(trackWidth / rightWheelDiameter);
		_rightDegPerDistance = (float)(360 / (Math.PI * rightWheelDiameter));
		// both
		_trackWidth = (float)trackWidth;
		_parity = (byte) (reverse ? -1 : 1);
		setTravelSpeed(.8f * getMaxTravelSpeed());
		setRotateSpeed(.8f * getMaxRotateSpeed());

		syncReports = new Object();
		unreportedMoves = 0;

	}

	
	public void stop(){
		stop(true);
	}
	/**
	 * Stops the NXT robot.
	 *  side effect: inform listeners of end of movement
	 * @param waitOnStop 
	 */
	public void stop(boolean waitOnStop)
	{
//		synchronized (syncReports) {
//			if(isMoving()) unreportedMoves++;

			_left.stop(true);
			_right.stop(true);
			System.out.println("Pilot wait for complete ...");
			waitComplete();
//			Thread.yield(); 
			if(waitOnStop)
				Delay.msDelay(1000); // give other threads a chance to react
//			try {
//				if(unreportedMoves > 0)
//					this.wait(3000); // Wait until every listener is called or too much time has passed
//				unreportedMoves = 0;
//			} catch (InterruptedException e) {
//				// TODO Auto-generated catch block
//				e.printStackTrace();
//			}
//		}
			System.out.println("Saliendo de stop");
	}

	/*
	 * Returns the left motor.
	 * @return left motor.
	 */
	/*public RegulatedMotor getLeft()
	  {
	    return _left;
	  }*/

	/*
	 * returns the right motor.
	 * @return right motor.
	 */
	/*public RegulatedMotor getRight()
	  {
	    return _right;
	  }*/

	/**
	 * Returns the tachoCount of the left motor
	 * @return tachoCount of left motor. Positive value means motor has moved
	 *         the robot forward.
	 */
	private int getLeftCount()
	{
		return _parity * _left.getTachoCount();
	}

	/**
	 * Returns the tachoCount of the right motor
	 * @return tachoCount of the right motor. Positive value means motor has
	 *         moved the robot forward.
	 */
	private int getRightCount()
	{
		return _parity * _right.getTachoCount();
	}

	/*
	 * Returns the actual speed of the left motor
	 * @return actual speed of left motor in degrees per second. A negative
	 *         value if motor is rotating backwards.
	 **/
	/*public int getLeftActualSpeed()
	  {
	    return _left.getRotationSpeed();
	  }*/

	/*
	 * Returns the actual speed of right motor
	 * @return actual speed of right motor in degrees per second. A negative
	 *         value if motor is rotating backwards.
	 **/
	/*public int getRightActualSpeed()
	  {
	    return _right.getRotationSpeed();
	  }*/

	private void setSpeed(final int leftSpeed, final int rightSpeed)
	{
		_left.setSpeed(leftSpeed);
		_right.setSpeed(rightSpeed);
	}

	/**
	 * set travel speed in wheel diameter units per second
	 * @param travelSpeed : speed in distance (wheel diameter)units/sec
	 */
	public void setTravelSpeed(final double travelSpeed)
	{
		_robotTravelSpeed = (float)travelSpeed;
		_motorSpeed = (int)Math.round(0.5 * travelSpeed * (_leftDegPerDistance + _rightDegPerDistance));
		setSpeed((int)Math.round(travelSpeed * _leftDegPerDistance), (int)Math.round(travelSpeed * _rightDegPerDistance));
	}

	public double getTravelSpeed()
	{
		return _robotTravelSpeed;
	}

	/**
	 * Sets the acceleration of both motors.
	 * @param accel
	 * @see lejos.robotics.RegulatedMotor#setAcceleration(int acceleration)
	 */
	public void setAcceleration(int accel)
	{
		_left.setAcceleration(accel);
		_right.setAcceleration(accel);

	}

	public double getMaxTravelSpeed()
	{
		return Math.min(_left.getMaxSpeed(), _right.getMaxSpeed()) / Math.max(_leftDegPerDistance, _rightDegPerDistance);
		// max degree/second divided by degree/unit = unit/second
	}

	/**
	 * sets the rotation speed of the vehicle, degrees per second
	 * @param rotateSpeed
	 */
	public void setRotateSpeed(double rotateSpeed)
	{
		_robotRotateSpeed = (float)rotateSpeed;
		setSpeed((int)Math.round(rotateSpeed * _leftTurnRatio), (int)Math.round(rotateSpeed * _rightTurnRatio));
	}


	public double getRotateSpeed()
	{
		return _robotRotateSpeed;
	}


	public float getMaxRotateSpeed()
	{
		return Math.min(_left.getMaxSpeed(), _right.getMaxSpeed()) / Math.max(_leftTurnRatio, _rightTurnRatio);
		// max degree/second divided by degree/unit = unit/second
	}


	public double getRotateMaxSpeed()
	{
		return getMaxRotateSpeed();
	}

	/**
	 * Starts the NXT robot moving forward.
	 */
	public void forward()
	{
		_type = Move.MoveType.TRAVEL;
		movementStart(false);
		setSpeed(Math.round(_robotTravelSpeed * _leftDegPerDistance), Math.round(_robotTravelSpeed * _rightDegPerDistance));
		if (_parity == 1)
		{
			fwd();
		} else
		{
			bak();
		}
	}

	/**
	 *  Starts the NXT robot moving backward.
	 */
	public void backward()
	{
		_type = Move.MoveType.TRAVEL;
		movementStart(false);
		setSpeed(Math.round(_robotTravelSpeed * _leftDegPerDistance), Math.round(_robotTravelSpeed * _rightDegPerDistance));

		if (_parity == 1)
		{
			bak();
		} else
		{
			fwd();
		}
	}

	/**
	 * Motors backward. This is called by forward() and backward(), demending in parity.
	 */
	private void bak()
	{
		_left.backward();
		_right.backward();
	}

	/**
	 * Motors forward. This is called by forward() and backward() depending in parity.
	 */
	private void fwd()
	{
		_left.forward();
		_right.forward();
	}

	public void rotateLeft()
	{
		_type = Move.MoveType.ROTATE;
		movementStart(true);
		if(_parity >0)
		{
			_right.forward();
			_left.backward();
		}
		else
		{
			_left.forward();
			_right.backward();
		}
	}

	public void rotateRight()
	{
		_type = Move.MoveType.ROTATE;
		movementStart(true);
		if (_parity > 0)
		{
			_left.forward();
			_right.backward();
		} else
		{
			_right.forward();
			_left.backward();
		}
	}

	/**
	 * Rotates the NXT robot through a specific angle. Returns when angle is
	 * reached. Wheels turn in opposite directions producing a zero radius turn.<br>
	 * Note: Requires correct values for wheel diameter and track width.
	 * calls rotate(angle,false)
	 * @param angle
	 *            The wanted angle of rotation in degrees. Positive angle rotate
	 *            left (clockwise), negative right.
	 */
	public void  rotate(final double angle)
	{
		rotate(angle, false);
	}

	/**
	 * Rotates the NXT robot through a specific angle. Returns when angle is
	 * reached. Wheels turn in opposite directions producing a zero radius turn.<br>
	 * Note: Requires correct values for wheel diameter and track width.
	 * Side effect: inform listeners
	 * @param angle
	 *            The wanted angle of rotation in degrees. Positive angle rotate
	 *            left (clockwise), negative right.
	 * @param immediateReturn
	 *            If true this method returns immediately.
	 */
	public void rotate(final double angle, final boolean immediateReturn)
	{
		_type = Move.MoveType.ROTATE;
		movementStart(immediateReturn);
		setSpeed(Math.round(_robotRotateSpeed * _leftTurnRatio), Math.round(_robotRotateSpeed * _rightTurnRatio));
		int rotateAngleLeft = _parity * (int) (angle * _leftTurnRatio);
		int rotateAngleRight = _parity * (int) (angle * _rightTurnRatio);
		_left.rotate(-rotateAngleLeft, true);
		_right.rotate(rotateAngleRight, immediateReturn);

		if (!immediateReturn)  while (isMoving()) Thread.yield();
	}

	/**
	 * Moves the NXT robot a specific distance in an (hopefully) straight line.<br>
	 * A positive distance causes forward motion, a negative distance moves
	 * backward. If a drift correction has been specified in the constructor it
	 * will be applied to the left motor.
	 * calls travel(distance, false)
	 * @param distance
	 *            The distance to move. Unit of measure for distance must be
	 *            same as wheelDiameter and trackWidth.
	 **/
	public void  travel(final double distance)
	{
		travel(distance, false);
	}

	/**
	 * Moves the NXT robot a specific distance in an (hopefully) straight line.<br>
	 * A positive distance causes forward motion, a negative distance moves
	 * backward. If a drift correction has been specified in the constructor it
	 * will be applied to the left motor.
	 *
	 * @param distance
	 *            The distance to move. Unit of measure for distance must be
	 *            same as wheelDiameter and trackWidth.
	 * @param immediateReturn
	 *            If true this method returns immediately.
	 */
	public void travel(final double distance, final boolean immediateReturn)
	{
		_type = Move.MoveType.TRAVEL;
		if (distance == Double.POSITIVE_INFINITY)
		{
			forward();
			return;
		}
		if ((distance == Double.NEGATIVE_INFINITY))
		{
			backward();
			return;
		}
		movementStart(immediateReturn);
		setSpeed(Math.round(_robotTravelSpeed * _leftDegPerDistance), Math.round(_robotTravelSpeed * _rightDegPerDistance));
		_left.rotate((int) (_parity * distance * _leftDegPerDistance), true);
		_right.rotate((int) (_parity * distance * _rightDegPerDistance),
				immediateReturn);
		if (!immediateReturn) waitComplete();
	}

	public void arcForward(final double radius)
	{
		_type = Move.MoveType.ARC;
		movementStart(true);
		double turnRate = turnRate(radius);
		steerPrep(turnRate); // sets motor speeds
		if(_parity >0)_outside.forward();
		else _outside.backward();
		if (_parity * _steerRatio > 0)  _inside.forward();
		else _inside.backward();
	}

	public void arcBackward(final double radius)
	{
		_type = Move.MoveType.ARC;
		movementStart(true);
		double turnRate = turnRate(radius);
		steerPrep(turnRate);// sets motor speeds
		if(_parity > 0)_outside.backward();
		else _outside.forward();
		if (_parity * _steerRatio > 0)_inside.backward();
		else _inside.forward();
	}

	public void arc(final double radius, final double angle)
	{
		arc(radius, angle, false);
	}

	public void  arc(final double radius, final double angle,
			final boolean immediateReturn)
	{
		if (radius == Double.POSITIVE_INFINITY || radius == Double.NEGATIVE_INFINITY)
		{
			forward();
			return;
		}
		steer(turnRate(radius), angle, immediateReturn);// type and move started called by steer()
		// if (!immediateReturn) waitComplete(); redundant I think - BB
	}

	public  void  travelArc(double radius, double distance)
	{
		travelArc(radius, distance, false);
	}

	public void travelArc(double radius, double distance, boolean immediateReturn)
	{
		if (radius == Double.POSITIVE_INFINITY || radius == Double.NEGATIVE_INFINITY)
		{
			travel(distance, immediateReturn);
			return;
		}
		//	    _type = Move.MoveType.ARC;
		//	    movementStart(immediateReturn);
		if (radius == 0)
		{
			throw new IllegalArgumentException("Zero arc radius");
		}
		double angle = (distance * 180) / ((float) Math.PI * radius);
		arc(radius, angle, immediateReturn);
	}

	/**
	 * Calculates the turn rate corresponding to the turn radius; <br>
	 * use as the parameter for steer() negative argument means center of turn
	 * is on right, so angle of turn is negative
	 * @param radius
	 * @return turnRate to be used in steer()
	 */
	private double turnRate(final double radius)
	{
		int direction;
		double radiusToUse;
		if (radius < 0)
		{
			direction = -1;
			radiusToUse = -radius;
		} else
		{
			direction = 1;
			radiusToUse = radius;
		}
		double ratio = (2 * radiusToUse - _trackWidth) / (2 * radiusToUse + _trackWidth);
		return (direction * 100 * (1 - ratio));
	}

	/**
	 * Starts the robot moving forward along a curved path. This method is similar to the
	 * {@link #arcForward(double radius )} method except it uses the <code> turnRate</code> parameter
	 * do determine the curvature of the path and therefore has the ability to drive straight. This makes
	 * it useful for line following applications.
	 * <p>
	 * The <code>turnRate</code> specifies the sharpness of the turn.  Use values  between -200 and +200.<br>
	 * A negative value means that center of the turn is on the left.  If the
	 * robot is traveling toward the top of the page the arc looks like this: <b>)</b>. <br>
	 * A positive value means that center of the turn is on the  right so the arc liiks  this: <b>(</b>. <br>.
	 *  In this class,  this parameter determines the  ratio of inner wheel speed to outer wheel speed <b>as a percent</b>.<br>
	 * <I>Formula:</I> <code>ratio = 100 - abs(turnRate)</code>.<br>
	 * When the ratio is negative, the outer and inner wheels rotate in
	 * opposite directions.
	 * Examples of how the formula works:
	 * <UL>
	 * <LI><code>steer(0)</code> -> inner and outer wheels turn at the same speed, travel  straight
	 * <LI><code>steer(25)</code> -> the inner wheel turns at 75% of the speed of the outer wheel, turn left
	 * <LI><code>steer(100)</code> -> the inner wheel stops and the outer wheel is at 100 percent, turn left
	 * <LI><code>steer(200)</code> -> the inner wheel turns at the same speed as the outer wheel - a zero radius turn.
	 * </UL>
	 * <p>
	 * Note: If you have specified a drift correction in the constructor it will not be applied in this method.
	 *
	 * @param turnRate If positive, the left side of the robot is on the inside of the turn. If negative,
	 * the left side is on the outside.
	 */
	public void steer(double turnRate)
	{
		if (turnRate == 0)
		{
			forward();
			return;
		}
		steerPrep(turnRate);
		if(_parity >0)_outside.forward();
		else _outside.backward();
		if (!_steering)  //only call movement start if this is the most recent methoc called
		{
			_type = Move.MoveType.ARC;
			movementStart(true);
			_steering = true;
		}
		if (_parity * _steerRatio > 0) _inside.forward();
		else _inside.backward();
	}

	/**
	 * Starts the robot moving backward  along a curved path. This method is essentially
	 * the same as
	 * {@link #steer(double turn rate )} except that the robot moves backward instead of forward.
	 * @param turnRate
	 */
	public void steerBackward(final double turnRate)
	{
		if (turnRate == 0)
		{
			if (_parity < 0)  forward();

			else backward();
			return;
		}
		steerPrep(turnRate);
		if(_parity > 0 ) _outside.backward();
		else _outside.forward();
		if (!_steering)  //only call movement start if this is the most recent methoc called
		{
			_type = Move.MoveType.ARC;
			movementStart(true);
			_steering = true;
		}
		if (_parity * _steerRatio > 0) _inside.backward();
		else _inside.forward();
	}

	/**
	 * Moves the robot along a curved path through a specified turn angle. This method is similar to the
	 * {@link #arc(double radius , double angle)} method except it uses a ratio of motor
	 * speeds to determine the curvature of the  path and therefore has the ability to drive straight. This makes
	 * it useful for line following applications. This method does not return until the robot has
	 * completed moving <code>angle</code> degrees along the arc.<br>
	 * The <code>turnRate</code> specifies the sharpness of the turn. Use values  between -200 and +200.<br>
	 * For details about how this parameter works.See {@link #steer(double turnRate) }
	 * <p>
	 * The robot will stop when its heading has changed by the amount of the  <code>angle</code> parameter.<br>
	 * If <code>angle</code> is positive, the robot will move in the direction that increases its heading (it turns left).<br>
	 * If <code>angle</code> is negative, the robot will move in the directin that decreases its heading (turns right).<br>
	 * If <code>angle</code> is zero, the robot will not move and the method returns immediately.
	 * <p> The sign of the turn rate and the sign of the angle together determine if the robot will move forward
	 * or backward. Assuming the robot is heading toward the top of the page. Then a positive turn rate means the
	 * arc looks like this: <b> )</b>  .  If the angle is positive, the robot moves forward to increase its heading angle.
	 * If negative,  it moves backward to decrease the heading.  <br> But if the turn rate is negative,
	 * the arc looks like this: <b> (  </b> .So a positive angle (increase in heading) means the robot
	 * moves backwards, while a negative angle means the robot moves forward to decrease
	 * its heading.
	 *
	 * <p>
	 * Note: If you have specified a drift correction in the constructor it will not be applied in this method.
	 *
	 * @param turnRate If positive, the left side of the robot is on the inside of the turn. If negative,
	 * the left side is on the outside.
	 * @param angle The angle through which the robot will rotate. If negative, the robot will move in the directin that decreases its heading.
	 */
	public void steer(final double turnRate, double angle)
	{
		steer(turnRate, angle, false);
	}

	/**
	 * Moves the robot along a curved path for a specified angle of rotation. This method is similar to the
	 * {@link #arc(double radius, double angle, boolean immediateReturn)} method except it uses the <code> turnRate()</code>
	 * parameter to determine the curvature of the path and therefore has the ability to drive straight.
	 * This makes it useful for line following applications. This method has the ability to return immediately
	 * by using the <code>immediateReturn</code> parameter set to <b>true</b>.
	 *
	 * <p>
	 * The <code>turnRate</code> specifies the sharpness of the turn. Use values between -200 and +200.<br>
	 * For details about how this parameter works, see {@link #steer(double turnRate) }
	 * <p>
	 * The robot will stop when its heading has changed by the amount of the  <code>angle</code> parameter.<br>
	 * If <code>angle</code> is positive, the robot will move in the direction that increases its heading (it turns left).<br>
	 * If <code>angle</code> is negative, the robot will move in the direction that decreases its heading (turns right).<br>
	 * If <code>angle</code> is zero, the robot will not move and the method returns immediately.<br>
	 * For more details about this parameter, see {@link #steer(double turnRate, double angle)}
	 * <p>
	 * Note: If you have specified a drift correction in the constructor it will not be applied in this method.
	 *
	 * @param turnRate If positive, the left side of the robot is on the inside of the turn. If negative,
	 * the left side is on the outside.
	 * @param angle The angle through which the robot will rotate. If negative, robot traces the turning circle backwards.
	 * @param immediateReturn If immediateReturn is true then the method returns immediately.
	 */
	public void steer(final double turnRate, final double angle,
			final boolean immediateReturn)
	{
		if (angle == 0)
		{
			return;
		}
		if (turnRate == 0)
		{
			forward();
			return;
		}
		_type = Move.MoveType.ARC;
		movementStart(immediateReturn);
		steerPrep(turnRate);
		int side = (int) Math.signum(turnRate);
		int rotAngle = (int) (angle * _trackWidth * 2 / (_leftWheelDiameter * (1 - _steerRatio)));
		_inside.rotate((int) (_parity * side * rotAngle * _steerRatio), true);
		_outside.rotate(_parity * side * rotAngle, immediateReturn);
		if (immediateReturn)
		{
			return;
		}
		waitComplete();
		_inside.setSpeed(_outside.getSpeed());
	}

	/**
	 * helper method used by steer(float) and steer(float,float,boolean)
	 * sets _outsideSpeed, _insideSpeed, _steerRatio
	 * @param turnRate
	 * @deprecated Access to this method will be private in NXJ version 1.0 when the CompassPilot is removed.
	 */
	@Deprecated protected void steerPrep(final double turnRate)
	{

		double rate = turnRate;
		if (rate < -200) rate = -200;
		if (rate > 200) rate = 200;

		if (turnRate < 0)
		{
			_inside = _right;
			_outside = _left;
			rate = -rate;
		} else
		{
			_inside = _left;
			_outside = _right;
		}
		_outside.setSpeed(_motorSpeed);
		_steerRatio = (float)(1 - rate / 100.0);
		_inside.setSpeed((int) (_motorSpeed * _steerRatio));
	}


	/**
	 * called by RegulatedMotor when a motor rotation is complete
	 * calls movementStop() after both motors stop;
	 * @param motor
	 * @param tachoCount
	 * @param  stall : true if motor is sealled
	 * @param ts  s time stamp
	 */
	public synchronized void rotationStopped(RegulatedMotor motor, int tachoCount, boolean stall,long ts)
	{
		if(motor.isStalled())stop();
		else if (!isMoving())movementStop();// a motor has stopped
	}

	/**
	 * MotorListener interface method is called by RegulatedMotor when a motor rotation starts.
	 * 
	 * @param motor
	 * @param tachoCount
	 * @param stall    true of the motor is stalled
	 * @param ts  time stamp
	 */
	public synchronized void rotationStarted(RegulatedMotor motor, int tachoCount, boolean stall,long ts)
	{ // Not used
	}

	/**
	 * called at start of a movement to inform the listeners  that a movement has started
	 * @deprecated Access to this method will be private in NXJ version 1.0 when the CompassPilot is removed.
	 */
	@Deprecated protected void movementStart(boolean alert)
	{
		if (isMoving())  movementStop();
		reset();
		for(MoveListener ml : _listeners)
			ml.moveStarted(new Move(_type,
					getMovementIncrement(), getAngleIncrement(), 
					_robotTravelSpeed, _robotRotateSpeed, isMoving()), this);
	}

	/**
	 * called by Arc() ,travel(),rotate(),stop() rotationStopped()
	 * calls moveStopped on listener
	 */
	private synchronized void movementStop()
	{
		for(MoveListener ml : _listeners){
			ml.moveStopped(new Move(_type,
					getMovementIncrement(), getAngleIncrement(), isMoving()), this);
		}

//		synchronized (syncReports) {
//			unreportedMoves--;
//			this.notifyAll();
//		}
	}

	/**
	 * @return true if the NXT robot is moving.
	 **/
	public boolean isMoving()
	{
//		System.out.println(String.valueOf(
//				"Motores " + _left.isMoving()) + " " + String.valueOf(_right.isMoving()));
		return _left.isMoving() || _right.isMoving();
	}

	/**
	 * wait for the current operation on both motors to complete
	 */
	private void waitComplete()
	{
		while(isMoving())
		{
			_left.waitComplete();
			_right.waitComplete();
//			Thread.yield();
		}
	}
	public boolean isStalled()
	{
		return _left.isStalled() || _right.isStalled();
	}
	/**
	 * Resets tacho count for both motors.
	 **/
	public void reset()
	{ 
		_leftTC = getLeftCount();
		_rightTC = getRightCount();
		_steering = false;
	}

	/**
	 * Set the radius of the minimum turning circle.
	 * Note: A DifferentialPilot robot can simulate a SteeringPilot robot by calling DifferentialPilot.setMinRadius()
	 * and setting the value to something greater than zero (example: 15 cm).
	 * 
	 * @param radius the radius in degrees
	 */
	public void setMinRadius(double radius)
	{
		_turnRadius = (float)radius;
	}

	public double getMinRadius()
	{
		return _turnRadius;
	}

	/**
	 * @deprecated Access to this method will be private in NXJ version 1.0 when the CompassPilot is removed
	 * and ArcNavigator sample will need to be updated to not use these methods.
	 * @return The move distance since it last started moving
	 */
	@Deprecated public float getMovementIncrement()
	{
		float left = (getLeftCount() - _leftTC)/ _leftDegPerDistance;
		float right = (getRightCount() - _rightTC) / _rightDegPerDistance;
		return /*_parity * */ (left + right) / 2.0f;
	}

	/**
	 * @deprecated Access to this method will be private in NXJ version 1.0 when the CompassPilot is removed
	 * and ArcNavigator sample will need to be updated to not use these methods.
	 * @return The angle rotated since rotation began.
	 * 
	 */
	@Deprecated public float getAngleIncrement()
	{
		return /*_parity * */(((getRightCount() - _rightTC) / _rightTurnRatio) -
				((getLeftCount()  - _leftTC) / _leftTurnRatio)) / 2.0f;
	}

	public void addMoveListener(MoveListener m)
	{
		_listeners.add(m);
	}

	public Move getMovement()
	{
		return  new Move(_type, getMovementIncrement(), getAngleIncrement(), isMoving());
	}

	private float _turnRadius = 0;
	/**
	 * Left motor.
	 * @deprecated Access to this field will be private in NXJ version 1.0 when the CompassPilot is removed.
	 */
	@Deprecated protected final RegulatedMotor _left;
	/**
	 * Right motor.
	 * @deprecated Access to this field will be private in NXJ version 1.0 when the CompassPilot is removed.
	 */
	@Deprecated protected final RegulatedMotor _right;
	/**
	 * The motor at the inside of the turn. set by steer(turnRate)
	 * used by other steer methodsl
	 */
	private RegulatedMotor _inside;
	/**
	 * The motor at the outside of the turn. set by steer(turnRate)
	 * used by other steer methodsl
	 * @deprecated Access to this field will be private in NXJ version 1.0 when the CompassPilot is removed.
	 */
	@Deprecated protected RegulatedMotor _outside;
	/**
	 * ratio of inside/outside motor speeds
	 * set by steer(turnRate)
	 * used by other steer methods;
	 */
	private float _steerRatio;
	private boolean _steering = false;
	/**
	 * Left motor degrees per unit of travel.
	 * @deprecated Access to this field will be private in NXJ version 1.0 when the CompassPilot is removed.
	 */
	@Deprecated protected final float _leftDegPerDistance;
	/**
	 * Right motor degrees per unit of travel.
	 * @deprecated Access to this field will be private in NXJ version 1.0 when the CompassPilot is removed.
	 */
	@Deprecated protected final float _rightDegPerDistance;
	/**
	 * Left motor revolutions for 360 degree rotation of robot (motors running
	 * in opposite directions). Calculated from wheel diameter and track width.
	 * Used by rotate() and steer() methods.
	 **/
	private final float _leftTurnRatio;
	/**
	 * Right motor revolutions for 360 degree rotation of robot (motors running
	 * in opposite directions). Calculated from wheel diameter and track width.
	 * Used by rotate() and steer() methods.
	 **/
	private final float _rightTurnRatio;
	/**
	 * Speed of robot for moving in wheel diameter units per seconds. Set by
	 * setSpeed(), setTravelSpeed()
	 */
	private float _robotTravelSpeed;
	/**
	 * Speed of robot for turning in degree per seconds.
	 */
	private float _robotRotateSpeed;
	/**
	 * Motor speed degrees per second. Used by forward(),backward() and steer().
	 */
	private int _motorSpeed;
	/**
	 * Motor rotation forward makes robot move forward if parity == 1.
	 */
	private byte _parity;
	/**
	 * Distance between wheels. Used in steer() and rotate().
	 */
	private final float _trackWidth;
	/**
	 * Diameter of left wheel.
	 */
	private final float _leftWheelDiameter;
	/**
	 * Diameter of right wheel.
	 */
	private final float _rightWheelDiameter;

	private  int _leftTC; // left tacho count
	private  int _rightTC; //right tacho count

	private ArrayList<MoveListener> _listeners= new ArrayList<MoveListener>();

	/**
	 * @deprecated Access to this field will be private in NXJ version 1.0 when the CompassPilot is removed.
	 */
	@Deprecated protected Move.MoveType _type;


}
